/*
** Copyright 2014-2017, 2021 Centreon
**
** Licensed under the Apache License, Version 2.0 (the "License");
** you may not use this file except in compliance with the License.
** You may obtain a copy of the License at
**
**     http://www.apache.org/licenses/LICENSE-2.0
**
** Unless required by applicable law or agreed to in writing, software
** distributed under the License is distributed on an "AS IS" BASIS,
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
** See the License for the specific language governing permissions and
** limitations under the License.
**
** For more information : contact@centreon.com
*/

#include "com/centreon/broker/bam/configuration/reader_v2.hh"

#include <fmt/format.h>

#include <cstring>
#include <deque>
#include <memory>
#include <unordered_map>

#include "com/centreon/broker/bam/configuration/reader_exception.hh"
#include "com/centreon/broker/bam/configuration/state.hh"
#include "com/centreon/broker/bam/dimension_ba_bv_relation_event.hh"
#include "com/centreon/broker/bam/dimension_ba_event.hh"
#include "com/centreon/broker/bam/dimension_ba_timeperiod_relation.hh"
#include "com/centreon/broker/bam/dimension_bv_event.hh"
#include "com/centreon/broker/bam/dimension_kpi_event.hh"
#include "com/centreon/broker/bam/dimension_timeperiod.hh"
#include "com/centreon/broker/bam/dimension_truncate_table_signal.hh"
#include "com/centreon/broker/config/applier/state.hh"
#include "com/centreon/broker/io/stream.hh"
#include "com/centreon/broker/log_v2.hh"
#include "com/centreon/broker/logging/logging.hh"
#include "com/centreon/broker/multiplexing/publisher.hh"
#include "com/centreon/broker/mysql.hh"
#include "com/centreon/broker/time/timeperiod.hh"
#include "com/centreon/exceptions/msg_fmt.hh"

using namespace com::centreon::exceptions;
using namespace com::centreon::broker;
using namespace com::centreon::broker::bam::configuration;

/**
 *  Constructor.
 *
 *  @param[in] centreon_db  Centreon database connection.
 *  @param[in] storage_cfg  Storage database configuration.
 */
reader_v2::reader_v2(mysql& centreon_db, const database_config& storage_cfg)
    : _mysql(centreon_db), _storage_cfg(storage_cfg) {}

/**
 *  Read configuration from database.
 *
 *  @param[out] st  All the configuration state for the BA subsystem
 *                  recuperated from the specified database.
 */
void reader_v2::read(state& st) {
  try {
    log_v2::bam()->info("loading dimensions.");
    _load_dimensions();
    log_v2::bam()->info("loading BAs.");
    _load(st.get_bas(), st.get_ba_svc_mapping());
    log_v2::bam()->info("loading KPIs.");
    _load(st.get_kpis());
    log_v2::bam()->info("loading boolean expressions.");
    _load(st.get_bool_exps());
    log_v2::bam()->info("loading mapping hosts <-> services.");
    _load(st.get_hst_svc_mapping());
    log_v2::bam()->info("bam configuration loaded.");
  } catch (std::exception const& e) {
    log_v2::bam()->error("Error while reading bam configuration: {}", e.what());
    st.clear();
    throw;
  }
}

/**
 *  Load KPIs from the DB.
 *
 *  @param[out] kpis The list of kpis in database.
 */
void reader_v2::_load(state::kpis& kpis) {
  try {
    std::string query(fmt::format(
        "SELECT  k.kpi_id, k.state_type, k.host_id, k.service_id, k.id_ba,"
        "        k.id_indicator_ba, k.meta_id, k.boolean_id,"
        "        k.current_status, k.last_level, k.downtime,"
        "        k.acknowledged, k.ignore_downtime,"
        "        k.ignore_acknowledged,"
        "        COALESCE(COALESCE(k.drop_warning, ww.impact), "
        "g.average_impact),"
        "        COALESCE(COALESCE(k.drop_critical, cc.impact), "
        "g.average_impact),"
        "        COALESCE(COALESCE(k.drop_unknown, uu.impact), "
        "g.average_impact),"
        "        k.last_state_change, k.in_downtime, k.last_impact"
        "  FROM mod_bam_kpi AS k"
        "  INNER JOIN mod_bam AS mb"
        "    ON k.id_ba = mb.ba_id"
        "  INNER JOIN mod_bam_poller_relations AS pr"
        "    ON pr.ba_id = mb.ba_id"
        "  LEFT JOIN mod_bam_impacts AS ww"
        "    ON k.drop_warning_impact_id = ww.id_impact"
        "  LEFT JOIN mod_bam_impacts AS cc"
        "    ON k.drop_critical_impact_id = cc.id_impact"
        "  LEFT JOIN mod_bam_impacts AS uu"
        "    ON k.drop_unknown_impact_id = uu.id_impact"
        "  LEFT JOIN (SELECT id_ba, 100.0 / COUNT(kpi_id) AS average_impact"
        "               FROM mod_bam_kpi"
        "               WHERE activate='1'"
        "               GROUP BY id_ba) AS g"
        "    ON k.id_ba=g.id_ba"
        "  WHERE k.activate='1'"
        "    AND mb.activate='1'"
        "    AND pr.poller_id={}",
        config::applier::state::instance().poller_id()));
    std::promise<database::mysql_result> promise;
    _mysql.run_query_and_get_result(query, &promise, 0);
    try {
      database::mysql_result res(promise.get_future().get());
      while (_mysql.fetch_row(res)) {
        // KPI object.
        uint32_t kpi_id(res.value_as_u32(0));
        kpis[kpi_id] = kpi(kpi_id,                 // ID.
                           res.value_as_i32(1),    // State type.
                           res.value_as_u32(2),    // Host ID.
                           res.value_as_u32(3),    // Service ID.
                           res.value_as_u32(4),    // BA ID.
                           res.value_as_u32(5),    // BA indicator ID.
                           res.value_as_u32(6),    // Meta-service ID.
                           res.value_as_u32(7),    // Boolean expression ID.
                           res.value_as_i32(8),    // Status.
                           res.value_as_i32(9),    // Last level.
                           res.value_as_f32(10),   // Downtimed.
                           res.value_as_f32(11),   // Acknowledged.
                           res.value_as_bool(12),  // Ignore downtime.
                           res.value_as_bool(13),  // Ignore acknowledgement.
                           res.value_as_f64(14),   // Warning.
                           res.value_as_f64(15),   // Critical.
                           res.value_as_f64(16));  // Unknown.

        // KPI state.
        if (!res.value_is_null(17)) {
          kpi_event e(kpi_id, res.value_as_u32(4), res.value_as_u64(17));
          e.status = res.value_as_i32(8);
          e.in_downtime = res.value_as_bool(18);
          e.impact_level = res.value_is_null(19) ? -1 : res.value_as_f64(19);
          kpis[kpi_id].set_opened_event(e);
        }
      }
    } catch (std::exception const& e) {
      throw msg_fmt("BAM: could not retrieve KPI configuration from DB: {}",
                    e.what());
    }

    // Load host ID/service ID of meta-services (temporary fix until
    // Centreon Broker 3 where meta-services will be computed by Broker
    // itself.
    for (state::kpis::iterator it = kpis.begin(), end = kpis.end(); it != end;
         ++it) {
      if (it->second.is_meta()) {
        std::string query(
            fmt::format("SELECT hsr.host_host_id, hsr.service_service_id"
                        "  FROM service AS s"
                        "  LEFT JOIN host_service_relation AS hsr"
                        "    ON s.service_id=hsr.service_service_id"
                        "  WHERE s.service_description='meta_{}'",
                        it->second.get_meta_id()));
        std::promise<database::mysql_result> promise;
        _mysql.run_query_and_get_result(query, &promise, 0);
        try {
          database::mysql_result res(promise.get_future().get());
          if (!_mysql.fetch_row(res))
            throw msg_fmt(
                "virtual service of meta-service {}"
                " does not exist",
                it->first);
          it->second.set_host_id(res.value_as_u32(0));
          it->second.set_service_id(res.value_as_u32(1));
        } catch (std::exception const& e) {
          throw msg_fmt("could not retrieve virtual meta-service's service: {}",
                        e.what());
        }
      }
    }
  } catch (reader_exception const& e) {
    (void)e;
    throw;
  } catch (std::exception const& e) {
    throw reader_exception(
        "BAM: could not retrieve KPI configuration from DB: {}", e.what());
  }
}

/**
 *  Load BAs from the DB.
 *
 *  @param[out] bas      The list of BAs in database.
 *  @param[out] mapping  The mapping of BA ID to host name/service
 *                       description.
 */
void reader_v2::_load(state::bas& bas, bam::ba_svc_mapping& mapping) {
  log_v2::bam()->info("BAM: loading BAs");
  try {
    {
      std::string query(
          fmt::format("SELECT b.ba_id, b.name, b.state_source, b.level_w,"
                      " b.level_c, b.last_state_change, b.current_status,"
                      " b.in_downtime, b.inherit_kpi_downtimes"
                      " FROM mod_bam AS b"
                      " INNER JOIN mod_bam_poller_relations AS pr"
                      " ON b.ba_id=pr.ba_id"
                      " WHERE b.activate='1'"
                      " AND pr.poller_id={}",
                      config::applier::state::instance().poller_id()));
      std::promise<database::mysql_result> promise;
      _mysql.run_query_and_get_result(query, &promise, 0);
      try {
        database::mysql_result res(promise.get_future().get());
        while (_mysql.fetch_row(res)) {
          // BA object.
          uint32_t ba_id(res.value_as_u32(0));
          bas[ba_id] = ba(ba_id,                // ID.
                          res.value_as_str(1),  // Name.
                          static_cast<configuration::ba::state_source>(
                              res.value_as_u32(2)),  // State source.
                          res.value_as_f32(3),       // Warning level.
                          res.value_as_f32(4),       // Critical level.
                          static_cast<configuration::ba::downtime_behaviour>(
                              res.value_as_u32(8)));  // Downtime inheritance.

          // BA state.
          if (!res.value_is_null(5)) {
            ba_event e;
            e.ba_id = ba_id;
            e.start_time = res.value_as_u64(5);
            e.status = res.value_as_i32(6);
            e.in_downtime = res.value_as_bool(7);
            bas[ba_id].set_opened_event(e);
            log_v2::bam()->trace(
                "BAM: ba {} configuration (start_time:{}, in downtime: {})",
                ba_id, e.start_time, e.in_downtime);
          }
        }
      } catch (std::exception const& e) {
        throw msg_fmt("BAM: {}", e.what());
      }
    }
  } catch (reader_exception const& e) {
    (void)e;
    throw;
  } catch (std::exception const& e) {
    throw reader_exception(
        "BAM: could not retrieve BA configuration from DB: {}", e.what());
  }

  // Load host_id/service_id of virtual BA services. All the associated
  // services have for description 'ba_[id]'.
  try {
    std::promise<database::mysql_result> promise;
    _mysql.run_query_and_get_result(
        "SELECT h.host_name, s.service_description,"
        "       hsr.host_host_id, hsr.service_service_id"
        "  FROM service AS s"
        "  INNER JOIN host_service_relation AS hsr"
        "    ON s.service_id=hsr.service_service_id"
        "  INNER JOIN host AS h"
        "    ON hsr.host_host_id=h.host_id"
        "  WHERE s.service_description LIKE 'ba_%'",
        &promise, 0);
    database::mysql_result res(promise.get_future().get());
    while (_mysql.fetch_row(res)) {
      uint32_t host_id = res.value_as_u32(2);
      uint32_t service_id = res.value_as_u32(3);
      std::string service_description = res.value_as_str(1);
      service_description.erase(0, strlen("ba_"));

      if (!service_description.empty()) {
        try {
          uint32_t ba_id = std::stoul(service_description);
          state::bas::iterator found = bas.find(ba_id);
          if (found == bas.end()) {
            log_v2::bam()->info(
                "BAM: virtual BA service '{}' of host '{}' references an "
                "unknown BA ({})",
                res.value_as_str(1), res.value_as_str(0), ba_id);
            continue;
          }
          found->second.set_host_id(host_id);
          found->second.set_service_id(service_id);
          mapping.set(ba_id, res.value_as_str(0), res.value_as_str(1));
        } catch (std::exception const& e) {
          log_v2::bam()->info(
              "BAM: service '{}' of host '{}' is not a valid virtual BA "
              "service",
              res.value_as_str(1), res.value_as_str(0));
          continue;
        }
      }
    }
  } catch (reader_exception const& e) {
    (void)e;
    throw;
  } catch (std::exception const& e) {
    throw reader_exception("BAM: could not retrieve BA service IDs from DB: {}",
                           e.what());
  }

  // Test for BA without service ID.
  for (state::bas::const_iterator it(bas.begin()), end(bas.end()); it != end;
       ++it)
    if (it->second.get_service_id() == 0)
      throw reader_exception("BAM: BA {} has no associated service",
                             it->second.get_id());
}

/**
 *  Load boolean expressions from the DB.
 *
 *  @param[out] bool_exps The list of bool expression in database.
 */
void reader_v2::_load(state::bool_exps& bool_exps) {
  // Load boolean expressions themselves.
  try {
    std::string query(
        fmt::format("SELECT b.boolean_id, b.name, b.expression, b.bool_state"
                    " FROM mod_bam_boolean AS b"
                    " INNER JOIN mod_bam_kpi AS k"
                    " ON b.boolean_id=k.boolean_id"
                    " INNER JOIN mod_bam_poller_relations AS pr"
                    " ON k.id_ba=pr.ba_id"
                    " WHERE b.activate=1"
                    " AND pr.poller_id={}",
                    config::applier::state::instance().poller_id()));
    std::promise<database::mysql_result> promise;
    _mysql.run_query_and_get_result(query, &promise, 0);
    database::mysql_result res(promise.get_future().get());
    while (_mysql.fetch_row(res)) {
      bool_exps[res.value_as_u32(0)] =
          bool_expression(res.value_as_u32(0),    // ID.
                          res.value_as_str(1),    // Name.
                          res.value_as_str(2),    // Expression.
                          res.value_as_bool(3));  // Impact if.
    }
  } catch (reader_exception const& e) {
    (void)e;
    throw;
  } catch (std::exception const& e) {
    throw reader_exception(
        "BAM: could not retrieve boolean expression "
        "configuration from DB: {}",
        e.what());
  }
}

/**
 *  Load host/service IDs from the DB.
 *
 *  @param[out] mapping  Host/service mapping.
 */
void reader_v2::_load(bam::hst_svc_mapping& mapping) {
  try {
    // XXX : expand hostgroups and servicegroups
    std::promise<database::mysql_result> promise;
    _mysql.run_query_and_get_result(
        "SELECT h.host_id, s.service_id, h.host_name, "
        "s.service_description,service_activate FROM "
        "service s LEFT JOIN host_service_relation hsr ON "
        "s.service_id=hsr.service_service_id "
        "LEFT JOIN host h ON hsr.host_host_id=h.host_id",
        &promise, 0);
    database::mysql_result res(promise.get_future().get());
    while (_mysql.fetch_row(res))
      mapping.set_service(res.value_as_str(2), res.value_as_str(3),
                          res.value_as_u32(0), res.value_as_u32(1),
                          res.value_as_str(4) == "1");
  } catch (reader_exception const& e) {
    (void)e;
    throw;
  } catch (std::exception const& e) {
    throw reader_exception("BAM: could not retrieve host/service IDs: {}",
                           e.what());
  }
}

/**
 *  Load the dimensions from the database.
 */
void reader_v2::_load_dimensions() {
  log_v2::bam()->trace("load dimensions");
  auto out{std::make_unique<multiplexing::publisher>()};
  // As this operation is destructive (it truncates the database),
  // we cache the data until we are sure we have all the data
  // needed from the database.
  std::deque<std::shared_ptr<io::data>> datas;
  datas.emplace_back(std::make_shared<dimension_truncate_table_signal>(true));

  // Load the dimensions.
  std::unordered_map<uint32_t, std::shared_ptr<dimension_ba_event>> bas;

  // Load the timeperiods themselves.
  std::promise<database::mysql_result> promise_tp;
  _mysql.run_query_and_get_result(
      "SELECT tp_id, tp_name, tp_sunday, tp_monday, tp_tuesday, "
      "tp_wednesday, tp_thursday, tp_friday, tp_saturday"
      " FROM timeperiod",
      &promise_tp, 0);

  // Load the BAs.
  std::string query_ba(
      fmt::format("SELECT b.ba_id, b.name, b.description,"
                  " b.sla_month_percent_warn, b.sla_month_percent_crit,"
                  " b.sla_month_duration_warn,"
                  " b.sla_month_duration_crit, b.id_reporting_period"
                  " FROM mod_bam AS b"
                  " INNER JOIN mod_bam_poller_relations AS pr"
                  " ON b.ba_id=pr.ba_id"
                  " WHERE b.activate='1'"
                  " AND pr.poller_id={}",
                  config::applier::state::instance().poller_id()));
  std::promise<database::mysql_result> promise_ba;
  _mysql.run_query_and_get_result(query_ba, &promise_ba, 0);

  // Load the BVs.
  std::promise<database::mysql_result> promise_bv;
  _mysql.run_query_and_get_result(
      "SELECT id_ba_group, ba_group_name, ba_group_description"
      " FROM mod_bam_ba_groups",
      &promise_bv, 0);

  // Load the BA BV relations.
  std::string query(
      fmt::format("SELECT id_ba, id_ba_group"
                  " FROM mod_bam_bagroup_ba_relation as r"
                  " INNER JOIN mod_bam AS b"
                  " ON b.ba_id = r.id_ba"
                  " INNER JOIN mod_bam_poller_relations AS pr"
                  " ON b.ba_id=pr.ba_id"
                  " WHERE b.activate='1'"
                  " AND pr.poller_id={}",
                  config::applier::state::instance().poller_id()));
  std::promise<database::mysql_result> promise_ba_bv;
  _mysql.run_query_and_get_result(query, &promise_ba_bv, 0);

  // Load the KPIs
  // Unfortunately, we need to get the names of the
  // service/host/meta_service/ba/boolean expression associated with
  // this KPI. This explains the numerous joins.
  std::string query_kpi{
      fmt::format("SELECT k.kpi_id, k.kpi_type, k.host_id, k.service_id,"
                  "       k.id_ba, k.id_indicator_ba, k.meta_id,"
                  "       k.boolean_id,"
                  "       COALESCE(COALESCE(k.drop_warning, ww.impact), "
                  "g.average_impact),"
                  "       COALESCE(COALESCE(k.drop_critical, cc.impact), "
                  "g.average_impact),"
                  "       COALESCE(COALESCE(k.drop_unknown, uu.impact), "
                  "g.average_impact),"
                  "       h.host_name, s.service_description, b.name,"
                  "       meta.meta_name, boo.name"
                  "  FROM mod_bam_kpi AS k"
                  "  LEFT JOIN mod_bam_impacts AS ww"
                  "    ON k.drop_warning_impact_id = ww.id_impact"
                  "  LEFT JOIN mod_bam_impacts AS cc"
                  "    ON k.drop_critical_impact_id = cc.id_impact"
                  "  LEFT JOIN mod_bam_impacts AS uu"
                  "    ON k.drop_unknown_impact_id = uu.id_impact"
                  "  LEFT JOIN host AS h"
                  "    ON h.host_id = k.host_id"
                  "  LEFT JOIN service AS s"
                  "    ON s.service_id = k.service_id"
                  "  INNER JOIN mod_bam AS b"
                  "    ON b.ba_id = k.id_ba"
                  "  INNER JOIN mod_bam_poller_relations AS pr"
                  "    ON b.ba_id = pr.ba_id"
                  "  LEFT JOIN meta_service AS meta"
                  "    ON meta.meta_id = k.meta_id"
                  "  LEFT JOIN mod_bam_boolean as boo"
                  "    ON boo.boolean_id = k.boolean_id"
                  "  LEFT JOIN (SELECT id_ba, 100.0 / COUNT(kpi_id) AS "
                  "average_impact"
                  "               FROM mod_bam_kpi"
                  "               WHERE activate='1'"
                  "               GROUP BY id_ba) AS g"
                  "   ON k.id_ba=g.id_ba"
                  "  WHERE k.activate='1'"
                  "    AND b.activate='1'"
                  "    AND pr.poller_id={}",
                  config::applier::state::instance().poller_id())};
  std::promise<database::mysql_result> promise_kpi;
  _mysql.run_query_and_get_result(query_kpi, &promise_kpi, 0);

  // Load the ba-timeperiods relations.
  std::promise<database::mysql_result> promise_ba_tp;
  _mysql.run_query_and_get_result(
      "SELECT ba_id, tp_id FROM mod_bam_relations_ba_timeperiods",
      &promise_ba_tp, 0);

  try {
    database::mysql_result res(promise_tp.get_future().get());
    while (_mysql.fetch_row(res)) {
      auto tp(std::make_shared<dimension_timeperiod>(res.value_as_u32(0),
                                                     res.value_as_str(1)));
      tp->sunday = res.value_as_str(2);
      tp->monday = res.value_as_str(3);
      tp->tuesday = res.value_as_str(4);
      tp->wednesday = res.value_as_str(5);
      tp->thursday = res.value_as_str(6);
      tp->friday = res.value_as_str(7);
      tp->saturday = res.value_as_str(8);
      datas.push_back(tp);
    }
  } catch (std::exception const& e) {
    throw reader_exception(
        "BAM: could not load some dimension table: "
        "could not load timeperiods from the database: {}",
        e.what());
  }

  try {
    database::mysql_result res(promise_ba.get_future().get());
    while (_mysql.fetch_row(res)) {
      auto ba{std::make_shared<dimension_ba_event>()};
      ba->ba_id = res.value_as_u32(0);
      ba->ba_name = res.value_as_str(1);
      ba->ba_description = res.value_as_str(2);
      ba->sla_month_percent_warn = res.value_as_f64(3);
      ba->sla_month_percent_crit = res.value_as_f64(4);
      ba->sla_duration_warn = res.value_as_i32(5);
      ba->sla_duration_crit = res.value_as_i32(6);
      datas.push_back(ba);
      bas[ba->ba_id] = ba;
      if (!res.value_is_null(7)) {
        std::shared_ptr<dimension_ba_timeperiod_relation> dbtr(
            new dimension_ba_timeperiod_relation);
        dbtr->ba_id = res.value_as_u32(0);
        dbtr->timeperiod_id = res.value_as_u32(7);
        dbtr->is_default = true;
        datas.push_back(dbtr);
      }
    }
  } catch (std::exception const& e) {
    throw reader_exception(
        "BAM: could not load some dimension table: "
        "could not retrieve BAs from the database: {}",
        e.what());
  }

  try {
    database::mysql_result res(promise_bv.get_future().get());
    while (_mysql.fetch_row(res)) {
      std::shared_ptr<dimension_bv_event> bv(new dimension_bv_event);
      bv->bv_id = res.value_as_u32(0);
      bv->bv_name = res.value_as_str(1);
      bv->bv_description = res.value_as_str(2);
      datas.push_back(bv);
    }
  } catch (std::exception const& e) {
    throw reader_exception(
        "BAM: could not load some dimension table: "
        "could not retrieve BVs from the database: {}",
        e.what());
  }

  try {
    database::mysql_result res(promise_ba_bv.get_future().get());
    while (_mysql.fetch_row(res)) {
      std::shared_ptr<dimension_ba_bv_relation_event> babv(
          new dimension_ba_bv_relation_event);
      babv->ba_id = res.value_as_u32(0);
      babv->bv_id = res.value_as_u32(1);
      datas.push_back(babv);
    }
  } catch (const std::exception& e) {
    throw reader_exception(
        "BAM: could not load some dimension table: "
        "could not retrieve BV memberships of BAs: {}",
        e.what());
  }

  try {
    database::mysql_result res(promise_kpi.get_future().get());

    while (_mysql.fetch_row(res)) {
      auto k{std::make_shared<dimension_kpi_event>(res.value_as_u32(0))};
      k->host_id = res.value_as_u32(2);
      k->service_id = res.value_as_u32(3);
      k->ba_id = res.value_as_u32(4);
      k->kpi_ba_id = res.value_as_u32(5);
      k->meta_service_id = res.value_as_u32(6);
      k->boolean_id = res.value_as_u32(7);
      k->impact_warning = res.value_as_f64(8);
      k->impact_critical = res.value_as_f64(9);
      k->impact_unknown = res.value_as_f64(10);
      k->host_name = res.value_as_str(11);
      k->service_description = res.value_as_str(12);
      k->ba_name = res.value_as_str(13);
      k->meta_service_name = res.value_as_str(14);
      k->boolean_name = res.value_as_str(15);

      // Resolve the id_indicator_ba.
      if (k->kpi_ba_id) {
        std::unordered_map<uint32_t,
                           std::shared_ptr<dimension_ba_event>>::const_iterator
            found = bas.find(k->kpi_ba_id);
        if (found == bas.end()) {
          logging::error(logging::high)
              << "BAM: could not retrieve BA " << k->kpi_ba_id
              << " used as KPI " << k->kpi_id
              << " in dimension table: ignoring this KPI";
          continue;
        }
        k->kpi_ba_name = found->second->ba_name;
      }
      datas.push_back(k);
    }
  } catch (std::exception const& e) {
    throw reader_exception(
        "BAM: could not load some dimension table: "
        "could not retrieve KPI dimensions: {}",
        e.what());
  }

  try {
    database::mysql_result res(promise_ba_tp.get_future().get());
    while (_mysql.fetch_row(res)) {
      std::shared_ptr<dimension_ba_timeperiod_relation> dbtr(
          new dimension_ba_timeperiod_relation);
      dbtr->ba_id = res.value_as_u32(0);
      dbtr->timeperiod_id = res.value_as_u32(1);
      dbtr->is_default = false;
      datas.push_back(dbtr);
    }
  } catch (std::exception const& e) {
    throw reader_exception(
        "BAM: could not load some dimension table: could not retrieve the "
        "timeperiods associated with the BAs: {}",
        e.what());
  }

  // End the update.
  datas.emplace_back(std::make_shared<dimension_truncate_table_signal>(false));

  // Write all the cached data to the publisher.
  for (auto& e : datas)
    out->write(e);
}
