/**
 * Copyright 2018-2024 Centreon
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * For more information : contact@centreon.com
 */

#ifndef CCB_LUA_MACRO_CACHE_HH
#define CCB_LUA_MACRO_CACHE_HH

#include "com/centreon/broker/bam/internal.hh"
#include "com/centreon/broker/lua/internal.hh"
#include "com/centreon/broker/neb/internal.hh"
#include "com/centreon/broker/persistent_cache.hh"

namespace com::centreon::broker::lua {

/**
 *  @class macro_cache macro_cache.hh "com/centreon/broker/lua/macro_cache.hh"
 *  @brief Data cache for Lua macro.
 */
class macro_cache {
 public:
  struct pb_host_group_member_group_id_poller_id_getter {
    using result_type = std::pair<uint64_t, uint64_t>;

    result_type operator()(
        const std::shared_ptr<neb::pb_host_group_member>& data) const {
      return std::make_pair(data->obj().hostgroup_id(),
                            data->obj().poller_id());
    }
  };

  struct pb_host_group_member_host_id_group_id_getter {
    using result_type = std::pair<uint64_t, uint64_t>;

    result_type operator()(
        const std::shared_ptr<neb::pb_host_group_member>& data) const {
      return std::make_pair(data->obj().host_id(), data->obj().hostgroup_id());
    }
  };

  using host_group_member_cont = boost::multi_index::multi_index_container<
      std::shared_ptr<neb::pb_host_group_member>,
      boost::multi_index::indexed_by<
          boost::multi_index::ordered_non_unique<
              pb_host_group_member_group_id_poller_id_getter>,
          boost::multi_index::ordered_unique<
              pb_host_group_member_host_id_group_id_getter>>>;

  struct pb_service_group_member_group_id_poller_id_getter {
    using result_type = std::pair<uint64_t, uint64_t>;

    result_type operator()(
        const std::shared_ptr<neb::pb_service_group_member>& data) const {
      return std::make_pair(data->obj().servicegroup_id(),
                            data->obj().poller_id());
    }
  };

  struct pb_service_group_member_host_service_id_group_id_getter {
    using result_type = std::tuple<uint64_t, uint64_t, uint64_t>;

    result_type operator()(
        const std::shared_ptr<neb::pb_service_group_member>& data) const {
      return std::make_tuple(data->obj().host_id(), data->obj().service_id(),
                             data->obj().servicegroup_id());
    }
  };

  using service_group_member_cont = boost::multi_index::multi_index_container<
      std::shared_ptr<neb::pb_service_group_member>,
      boost::multi_index::indexed_by<
          boost::multi_index::ordered_non_unique<
              pb_service_group_member_group_id_poller_id_getter>,
          boost::multi_index::ordered_unique<
              pb_service_group_member_host_service_id_group_id_getter>>>;

 private:
  std::shared_ptr<persistent_cache> _cache;
  absl::flat_hash_map<uint64_t, std::shared_ptr<neb::pb_instance>> _instances;
  absl::flat_hash_map<uint64_t, std::shared_ptr<neb::pb_host>> _hosts;

  absl::btree_map<std::pair<uint64_t /*group id*/, uint64_t /*poller id*/>,
                  std::shared_ptr<neb::pb_host_group>>
      _host_groups;

  host_group_member_cont _host_group_members;

  absl::flat_hash_map<std::pair<uint64_t, uint64_t>,
                      std::shared_ptr<neb::pb_custom_variable>>
      _custom_vars;
  absl::flat_hash_map<std::pair<uint64_t, uint64_t>,
                      std::shared_ptr<neb::pb_service>>
      _services;
  /* The service groups cache stores also a set with the pollers telling they
   * need the cache. So if no more poller needs a service group, we can remove
   * it from the cache. */
  absl::btree_map<std::pair<uint64_t /*group id*/, uint64_t /*poller id*/>,
                  std::shared_ptr<neb::pb_service_group>>
      _service_groups;
  service_group_member_cont _service_group_members;
  absl::flat_hash_map<uint64_t, std::shared_ptr<storage::pb_index_mapping>>
      _index_mappings;
  absl::flat_hash_map<uint64_t, std::shared_ptr<storage::pb_metric_mapping>>
      _metric_mappings;
  absl::flat_hash_map<uint64_t, std::shared_ptr<bam::pb_dimension_ba_event>>
      _dimension_ba_events;
  std::unordered_multimap<
      uint64_t,
      std::shared_ptr<bam::pb_dimension_ba_bv_relation_event>>
      _dimension_ba_bv_relation_events;
  absl::flat_hash_map<uint64_t, std::shared_ptr<bam::pb_dimension_bv_event>>
      _dimension_bv_events;

 public:
  macro_cache(const std::shared_ptr<persistent_cache>& cache);
  macro_cache(const macro_cache&) = delete;
  ~macro_cache();

  void write(std::shared_ptr<io::data> const& data);

  const storage::pb_index_mapping& get_index_mapping(uint64_t index_id) const;
  const std::shared_ptr<storage::pb_metric_mapping>& get_metric_mapping(
      uint64_t metric_id) const;
  const std::shared_ptr<neb::pb_host>& get_host(uint64_t host_id) const;
  const std::shared_ptr<neb::pb_service>& get_service(
      uint64_t host_id,
      uint64_t service_id) const;
  const std::string& get_host_name(uint64_t host_id) const;
  const std::string& get_notes_url(uint64_t host_id, uint64_t service_id) const;
  const std::string& get_notes(uint64_t host_id, uint64_t service_id) const;
  const std::string& get_action_url(uint64_t host_id,
                                    uint64_t service_id) const;
  int32_t get_severity(uint64_t host_id, uint64_t service_id) const;
  std::string_view get_check_command(uint64_t host_id,
                                     uint64_t service_id = 0) const;
  const std::string& get_host_group_name(uint64_t id) const;
  const std::string& get_host_group_alias(uint64_t id) const;
  const host_group_member_cont& get_host_group_members() const {
    return _host_group_members;
  }
  const std::string& get_service_description(uint64_t host_id,
                                             uint64_t service_id) const;
  const std::string& get_service_group_name(uint64_t id) const;
  const service_group_member_cont& get_service_group_members() const {
    return _service_group_members;
  }
  const std::string& get_instance(uint64_t instance_id) const;

  const std::unordered_multimap<
      uint64_t,
      std::shared_ptr<bam::pb_dimension_ba_bv_relation_event>>&
  get_dimension_ba_bv_relation_events() const;
  const std::shared_ptr<bam::pb_dimension_ba_event>& get_dimension_ba_event(
      uint64_t id) const;
  const std::shared_ptr<bam::pb_dimension_bv_event>& get_dimension_bv_event(
      uint64_t id) const;

 private:
  macro_cache& operator=(macro_cache const& f);

  void _process_pb_instance(std::shared_ptr<io::data> const& data);
  void _process_pb_host(std::shared_ptr<io::data> const& data);
  void _process_pb_host_status(std::shared_ptr<io::data> const& data);
  void _process_pb_adaptive_host_status(const std::shared_ptr<io::data>& data);
  void _process_pb_adaptive_host(std::shared_ptr<io::data> const& data);
  void _process_pb_host_group(std::shared_ptr<io::data> const& data);
  void _process_pb_host_group_member(std::shared_ptr<io::data> const& data);
  void _process_pb_custom_variable(std::shared_ptr<io::data> const& data);
  void _process_pb_service(std::shared_ptr<io::data> const& data);
  void _process_pb_service_status(const std::shared_ptr<io::data>& data);
  void _process_pb_adaptive_service_status(
      const std::shared_ptr<io::data>& data);
  void _process_pb_adaptive_service(std::shared_ptr<io::data> const& data);
  void _process_pb_service_group(std::shared_ptr<io::data> const& data);
  void _process_pb_service_group_member(std::shared_ptr<io::data> const& data);
  void _process_index_mapping(std::shared_ptr<io::data> const& data);
  void _process_metric_mapping(std::shared_ptr<io::data> const& data);
  void _process_dimension_ba_event(std::shared_ptr<io::data> const& data);
  void _process_dimension_ba_bv_relation_event(
      std::shared_ptr<io::data> const& data);
  void _process_dimension_bv_event(std::shared_ptr<io::data> const& data);
  void _process_pb_dimension_truncate_table_signal(
      std::shared_ptr<io::data> const& data);

  void _save_to_disk();
};
}  // namespace com::centreon::broker::lua

#endif  // !CCB_LUA_MACRO_CACHE_HH
