<?php

/*
 * Copyright 2005 - 2023 Centreon (https://www.centreon.com/)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * For more information : contact@centreon.com
 *
 */

declare(strict_types=1);

namespace Core\Infrastructure\RealTime\Repository\Acknowledgement;

use Centreon\Infrastructure\DatabaseConnection;
use Centreon\Infrastructure\Repository\AbstractRepositoryDRB;
use Core\Application\RealTime\Repository\ReadAcknowledgementRepositoryInterface;
use Core\Domain\RealTime\Model\Acknowledgement;

class DbReadAcknowledgementRepository extends AbstractRepositoryDRB implements ReadAcknowledgementRepositoryInterface
{
    /**
     * @param DatabaseConnection $db
     */
    public function __construct(DatabaseConnection $db)
    {
        $this->db = $db;
    }

    /**
     * @inheritDoc
     */
    public function findOnGoingAcknowledgementByHostId(int $hostId): ?Acknowledgement
    {
        return $this->findOnGoingAcknowledegement($hostId, 0);
    }

    /**
     * @inheritDoc
     */
    public function findOnGoingAcknowledgementByHostIdAndServiceId(int $hostId, int $serviceId): ?Acknowledgement
    {
        return $this->findOnGoingAcknowledegement($hostId, $serviceId);
    }

    /**
     * @param int $hostId
     * @param int $serviceId
     *
     * @return Acknowledgement|null
     */
    private function findOnGoingAcknowledegement(int $hostId, int $serviceId): ?Acknowledgement
    {
        $acknowledgement = null;
        $sql = 'SELECT 1 AS REALTIME, ack.*, contact.contact_id AS author_id
                FROM `:dbstg`.acknowledgements ack
                LEFT JOIN `:db`.contact ON contact.contact_alias = ack.author
                INNER JOIN (
                    SELECT MAX(acknowledgement_id) AS acknowledgement_id
                    FROM `:dbstg`.acknowledgements ack
                    WHERE ack.host_id = :hostId
                    AND ack.service_id = :serviceId
                    AND ack.deletion_time IS NULL
                ) ack_max ON ack_max.acknowledgement_id = ack.acknowledgement_id';

        $statement = $this->db->prepare($this->translateDbName($sql));
        $statement->bindValue(':hostId', $hostId, \PDO::PARAM_INT);
        $statement->bindValue(':serviceId', $serviceId, \PDO::PARAM_INT);
        $statement->execute();

        if ($row = $statement->fetch(\PDO::FETCH_ASSOC)) {
            /** @var array<string,int|string|null> $row */
            return DbAcknowledgementFactory::createFromRecord($row);
        }

        return $acknowledgement;
    }
}
