import { equals, flatten, includes, map, pipe, pluck } from 'ramda';

import { buildListingEndpoint } from '@centreon/ui';

import { formatStatusFilter } from '../Listing/utils';
import { DisplayType } from '../Listing/models';
import { Resource } from '../../../models';

export const resourcesEndpoint = '/monitoring/resources';
export const viewByHostEndpoint = '/monitoring/resources/hosts';

interface BuildResourcesEndpointProps {
  limit: number;
  page: number;
  resources: Array<Resource>;
  sort;
  states: Array<string>;
  statuses: Array<string>;
  type: DisplayType;
}

const resourceTypesCustomParameters = [
  'host-group',
  'host-category',
  'service-group',
  'service-category'
];
const resourceTypesSearchParameters = ['host', 'service'];

const categories = ['host-category', 'service-category'];

const resourcesSearchMapping = {
  host: 'parent_name',
  service: 'name'
};

export const buildResourcesEndpoint = ({
  type,
  statuses,
  states,
  sort,
  limit,
  resources,
  page
}: BuildResourcesEndpointProps): string => {
  const baseEndpoint = equals(type, 'host')
    ? viewByHostEndpoint
    : resourcesEndpoint;

  const formattedType = equals(type, 'all') ? ['host', 'service'] : [type];
  const formattedStatuses = pipe(
    map((state) => formatStatusFilter(state)),
    flatten,
    map((state) => state.toLocaleUpperCase())
  )(statuses);

  const resourcesToApplyToCustomParameters = resources.filter(
    ({ resourceType }) => includes(resourceType, resourceTypesCustomParameters)
  );
  const resourcesToApplyToSearchParameters = resources.filter(
    ({ resourceType }) => includes(resourceType, resourceTypesSearchParameters)
  );

  const searchConditions = resourcesToApplyToSearchParameters.map(
    ({ resourceType, resources: resourcesToApply }) => {
      return resourcesToApply.map((resource) => ({
        field: resourcesSearchMapping[resourceType],
        values: {
          $rg: `^${resource.name}$`
        }
      }));
    }
  );

  return buildListingEndpoint({
    baseEndpoint,
    customQueryParameters: [
      { name: 'types', value: formattedType },
      { name: 'statuses', value: formattedStatuses },
      { name: 'states', value: states },
      ...resourcesToApplyToCustomParameters.map(
        ({ resourceType, resources: resourcesToApply }) => ({
          name: includes(resourceType, categories)
            ? `${resourceType.replace('-', '_')}_names`
            : `${resourceType.replace('-', '')}_names`,
          value: pluck('name', resourcesToApply)
        })
      )
    ],
    parameters: {
      limit,
      page,
      search: {
        conditions: flatten(searchConditions)
      },
      sort
    }
  });
};
