/* eslint-disable @typescript-eslint/no-var-requires */
const { defineConfig } = require('cypress');
const { devServer } = require('cypress-rspack-dev-server');
const {
  addMatchImageSnapshotPlugin
} = require('@simonsmith/cypress-image-snapshot/plugin');
const cypressCodeCoverageTask = require('@cypress/code-coverage/task');

const fs = require('fs');
const path = require('path');

module.exports = ({
  webpackConfig,
  cypressFolder,
  specPattern,
  env,
  useVite = false,
  excludeSpecPattern
}) => {
  const mainCypressFolder = cypressFolder || 'cypress';

  return defineConfig({
    component: {
      devServer: useVite
        ? {
            bundler: 'vite',
            framework: 'react',
            webpackConfig
          }
        : (devServerConfig) =>
            devServer({
              ...devServerConfig,
              framework: 'react',
              rspackConfig: webpackConfig
            }),
      excludeSpecPattern,
      setupNodeEvents: (on, config) => {
        addMatchImageSnapshotPlugin(on, config);

        cypressCodeCoverageTask(on, config);
        on('task', {
          coverageReport: () => {
            return null;
          }
        });

        on('before:browser:launch', (browser, launchOptions) => {
          if (browser.name === 'chrome' && browser.isHeadless) {
            launchOptions.args.push('--headless=new');
            launchOptions.args.push('--force-color-profile=srgb');
            launchOptions.args.push('--window-size=1400,1200');
            launchOptions.args.push('--max-old-space-size=4096');
            launchOptions.args.push('--disable-dev-shm-usage');
            launchOptions.args.push('--disable-gpu');
            launchOptions.args.push('--no-sandbox');
          }

          return launchOptions;
        });

        on('after:run', (results) => {
          const testRetries = {};
          if ('runs' in results) {
            results.runs.forEach((run) => {
              run.tests.forEach((test) => {
                if (test.attempts && test.attempts.length > 1 && test.state === 'passed') {
                  const testTitle = test.title.join(' > '); // Convert the array to a string
                  testRetries[testTitle] = test.attempts.length - 1;
                }
              });
            });
          }

          // Save the testRetries object to a file in the e2e/results directory
          const resultFilePath = path.join(
            mainCypressFolder,
            'results',
            'retries.json'
          );

          fs.writeFileSync(resultFilePath, JSON.stringify(testRetries, null, 2));
        });

        on('after:spec', () => {
          if (global.__coverage__) {
            delete global.__coverage__;
          }
        });
      },
      specPattern,
      supportFile: `${mainCypressFolder}/support/component.tsx`
    },
    env: {
      baseUrl: 'http://localhost:9092',
      codeCoverage: {
        exclude: [
          'cypress/**/*.*',
          'packages/**',
          'node_modules',
          '**/*.js',
          '**/*.spec.tsx'
        ]
      },
      ...env
    },
    reporter: 'mochawesome',
    reporterOptions: {
      html: false,
      json: true,
      overwrite: true,
      reportDir: `${mainCypressFolder}/results`,
      reportFilename: '[name]-report.json'
    },
    retries: {
      openMode: 0,
      runMode: 2
    },
    video: true,
    videosFolder: `${mainCypressFolder}/results/videos`,
    viewportHeight: 590,
    viewportWidth: 1280
  });
};
