<?php
/*
 * Copyright 2005-2019 Centreon
 * Centreon is developed by : Julien Mathis and Romain Le Merlus under
 * GPL Licence 2.0.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation ; either version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, see <http://www.gnu.org/licenses>.
 *
 * Linking this program statically or dynamically with other modules is making a
 * combined work based on this program. Thus, the terms and conditions of the GNU
 * General Public License cover the whole combination.
 *
 * As a special exception, the copyright holders of this program give Centreon
 * permission to link this program with independent modules to produce an executable,
 * regardless of the license terms of these independent modules, and to copy and
 * distribute the resulting executable under terms of Centreon choice, provided that
 * Centreon also meet, for each linked independent module, the terms  and conditions
 * of the license of that module. An independent module is a module which is not
 * derived from this program. If you modify this program, you may extend this
 * exception to your version of the program, but you are not obliged to do so. If you
 * do not wish to do so, delete this exception statement from your version.
 *
 * For more information : contact@centreon.com
 *
 *
 */

namespace Centreon\Application\Validation\Validator;

use App\Kernel;
use Centreon\ServiceProvider;
use LogicException;
use Psr\Container\ContainerInterface;
use Symfony\Component\DependencyInjection\Exception\ServiceCircularReferenceException;
use Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Centreon\Application\Validation\Constraints\UniqueEntity;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;
use Symfony\Component\Validator\Exception\ConstraintDefinitionException;
use Centreon\Application\Validation\Validator\Interfaces\CentreonValidatorInterface;

use function is_array;
use function is_string;

class UniqueEntityValidator extends ConstraintValidator implements CentreonValidatorInterface
{

    /**
     * @param $entity
     * @param Constraint $constraint
     *
     * @return void|null
     * @throws ConstraintDefinitionException
     * @throws UnexpectedTypeException
     * @throws LogicException
     * @throws ServiceCircularReferenceException
     * @throws ServiceNotFoundException
     */
    public function validate($entity, Constraint $constraint)
    {
        if (!$constraint instanceof UniqueEntity) {
            throw new UnexpectedTypeException($constraint, __NAMESPACE__ . '\UniqueEntity');
        }

        if (! is_array($constraint->fields) && ! is_string($constraint->fields)) {
            throw new UnexpectedTypeException($constraint->fields, 'array');
        }

        if (null !== $constraint->errorPath && ! is_string($constraint->errorPath)) {
            throw new UnexpectedTypeException($constraint->errorPath, 'string or null');
        }

        //define fields to check
        $fields = (array) $constraint->fields;
        $methodRepository = $constraint->repositoryMethod;
        $methodIdGetter = $constraint->entityIdentificatorMethod;

        if ([] === $fields) {
            throw new ConstraintDefinitionException('At least one field has to be specified.');
        } elseif (null === $entity) {
            return null;
        }

        foreach ($fields as $field) {
            $methodValueGetter = 'get' . ucfirst($field);
            $value = $entity->$methodValueGetter();

            $repository = (Kernel::createForWeb())
                ->getContainer()
                ->get($constraint->repository);

            $result = $repository->$methodRepository([$field => $value]);

            if ($result && $result->$methodIdGetter() !== $entity->$methodIdGetter()) {
                $this->context->buildViolation($constraint->message)
                    ->atPath($field)
                    ->setInvalidValue($value)
                    ->setCode(UniqueEntity::NOT_UNIQUE_ERROR)
                    ->setCause($result)
                    ->addViolation();
            }
        }
    }

    /**
     * List of required services
     *
     * @return array
     */
    public static function dependencies(): array
    {
        return [
            ServiceProvider::CENTREON_DB_MANAGER,
        ];
    }
}
