import {
  NumberField,
  SingleConnectedAutocompleteField,
  TextField,
  buildListingEndpoint
} from '@centreon/ui';
import { Box } from '@mui/material';
import { useTranslation } from 'react-i18next';
import { hostsConfigurationEndpoint } from '../../api/endpoints';
import { HostConfiguration as HostConfigurationModel } from '../../models';

import RedirectToTokensPage from '../RedirectToTokensPage';
import { useHostConfiguration } from './useHostConfiguration';

import CMATokens from './CMATokenField';

import {
  labelCACommonName,
  labelCaCertificate,
  labelDNSIP,
  labelPort,
  labelSelectHost
} from '../../translatedLabels';

interface Props {
  index: number;
  host: HostConfigurationModel;
}

const HostConfiguration = ({ index, host }: Props): JSX.Element => {
  const { t } = useTranslation();
  const {
    selectHost,
    changeAddress,
    hostErrors,
    hostTouched,
    changePort,
    areCertificateFieldsVisible,
    changeStringInput,
    changeCMAToken,
    token
  } = useHostConfiguration({
    index
  });

  return (
    <Box
      sx={{
        display: 'grid',
        gridTemplateColumns: 'repeat(2, 1fr)',
        gap: 2,
        width: 'calc(100% - 24px)'
      }}
    >
      <SingleConnectedAutocompleteField
        required
        label={t(labelSelectHost)}
        onChange={selectHost}
        getEndpoint={(parameters) =>
          buildListingEndpoint({
            baseEndpoint: hostsConfigurationEndpoint,
            parameters
          })
        }
        value={{ id: host.id, name: host.name }}
      />
      <TextField
        required
        value={host.address}
        onChange={changeAddress}
        label={t(labelDNSIP)}
        dataTestId={labelDNSIP}
        fullWidth
        inputProps={{
          'aria-label': labelDNSIP
        }}
        error={hostTouched?.address && hostErrors?.address}
      />
      <NumberField
        required
        value={host.port.toString()}
        onChange={changePort}
        label={t(labelPort)}
        dataTestId={labelPort}
        fullWidth
        error={hostTouched?.port && hostErrors?.port}
        inputProps={{
          min: 1,
          max: 65535
        }}
      />

      {areCertificateFieldsVisible && (
        <>
          <TextField
            value={host?.pollerCaCertificate || ''}
            onChange={changeStringInput('pollerCaCertificate')}
            label={t(labelCaCertificate)}
            dataTestId={labelCaCertificate}
            inputProps={{
              'aria-label': labelCaCertificate
            }}
            fullWidth
            error={
              (hostTouched?.pollerCaCertificate &&
                hostErrors?.pollerCaCertificate) ||
              undefined
            }
          />

          <TextField
            value={host?.pollerCaName || ''}
            onChange={changeStringInput('pollerCaName')}
            label={t(labelCACommonName)}
            inputProps={{
              'aria-label': labelCACommonName
            }}
            dataTestId={labelCACommonName}
            fullWidth
            error={
              (hostTouched?.pollerCaName && hostErrors?.pollerCaName) ||
              undefined
            }
          />

          <CMATokens changeCMAToken={changeCMAToken} value={token} />
          <div />
          <RedirectToTokensPage />
        </>
      )}
    </Box>
  );
};

export default HostConfiguration;
