<?php

/*
 * Copyright 2005 - 2025 Centreon (https://www.centreon.com/)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * For more information : contact@centreon.com
 *
 */

/**
 * Class
 *
 * @class CentreonCfgWriter
 * @description Class that allows to write Nagios configuration files
 */
class CentreonCfgWriter
{
    /** @var string */
    private $buffer = '';

    /** @var CentreonXML */
    private $xmlBuffer;

    /** @var Centreon */
    private $centreon;

    /** @var string */
    private $file_path;

    /**
     *  CentreonCfgWriter constructor
     *
     * @param Centreon $centreon
     * @param string $file_full_path
     * @return void
     */
    public function __construct($centreon, $file_full_path)
    {
        $this->centreon = $centreon;
        $this->xmlBuffer = new CentreonXML();
        $this->file_path = $file_full_path;
    }

    /**
     *  Defines cfg type
     *
     * @param string $type
     * @return void
     */
    public function startCfg($type): void
    {
        $this->writeText('define ' . $type . "{\n");
        $this->xmlBuffer->startElement($type);
    }

    /**
     *  Ends cfg
     *
     * @return void
     */
    public function endCfg(): void
    {
        $this->writeText("\t}\n\n");
        $this->xmlBuffer->endElement();
    }

    /**
     *  Sets attributes
     *
     * @param string $key
     * @param string $value
     * @return void
     */
    public function attribute($key, $value): void
    {
        $len = strlen($key);
        if ($len <= 9) {
            $this->writeText("\t" . $key . "\t\t\t\t" . $value . "\n");
        } elseif ($len > 9 && $len <= 18) {
            $this->writeText("\t" . $key . "\t\t\t" . $value . "\n");
        } elseif ($len >= 19 && $len <= 27) {
            $this->writeText("\t" . $key . "\t\t" . $value . "\n");
        } elseif ($len > 27) {
            $this->writeText("\t" . $key . "\t" . $value . "\n");
        }
        $this->xmlBuffer->writeElement($key, $value);
    }

    /**
     * Writes in file
     *
     * @return void
     */
    public function createCfgFile(): void
    {
        file_put_contents($this->file_path, $this->buffer);
    }

    /**
     * Returns XML format
     *
     * @return CentreonXML
     */
    public function getXML()
    {
        return $this->xmlBuffer;
    }

    /**
     *  Creates the file
     *
     * @return void
     */
    protected function createFile()
    {
        $this->createFileHeader();
    }

    /**
     *  Writes basic text line to buffer
     *  Returns the length of written text
     *
     * @param string $text
     *
     * @return int
     */
    protected function writeText($text)
    {
        $this->buffer .= $text;

        return strlen($text);
    }

    /**
     *  Inserts Header of the file
     *
     * @return void
     */
    protected function createFileHeader()
    {
        $time = date('F j, Y, g:i a');
        $by = $this->centreon->user->get_name();
        $len = $this->writeText("###################################################################\n");
        $this->writeText("#                                                                 #\n");
        $this->writeText("#                       GENERATED BY CENTREON                     #\n");
        $this->writeText("#                                                                 #\n");
        $this->writeText("#               Developped by :                                   #\n");
        $this->writeText("#                   - Julien Mathis                               #\n");
        $this->writeText("#                   - Romain Le Merlus                            #\n");
        $this->writeText("#                                                                 #\n");
        $this->writeText("#                           www.centreon.com                      #\n");
        $this->writeText("#                For information : contact@centreon.com           #\n");
        $this->writeText("###################################################################\n");
        $this->writeText("#                                                                 #\n");
        $this->writeText('#         Last modification ' . $time);

        $margin = strlen($time);
        $margin = $len - 28 - $margin - 2;

        for ($i = 0; $i != $margin; $i++) {
            $this->writeText(' ');
        }

        $this->writeText("#\n");
        $this->writeText('#         By ' . $by);
        $margin = $len - 13 - strlen($by) - 2;

        for ($i = 0; $i != $margin; $i++) {
            $this->writeText(' ');
        }
        $this->writeText("#\n");
        $this->writeText("#                                                                 #\n");
        $this->writeText("###################################################################\n\n");
    }
}
