import {
  all,
  equals,
  flatten,
  gt,
  isEmpty,
  length,
  pipe,
  pluck,
  type,
  uniq,
  uniqBy
} from 'ramda';

import {
  ListingModel,
  SelectEntry,
  buildListingEndpoint,
  resourceTypeQueryParameter,
  useFetchQuery
} from '@centreon/ui';

import { Metric, ServiceMetric, WidgetDataResource } from '../../../models';

import { serviceMetricsDecoder } from '../../../api/decoders';
import { metricsEndpoint } from '../../../api/endpoints';
import { buildResourceTypeNameForSearchParameter } from '../utils';

interface Props {
  resources: Array<WidgetDataResource>;
  selectedMetrics?: Array<Metric>;
}

interface UseListMetricsState {
  hasMultipleUnitsSelected: boolean;
  hasTooManyMetrics: boolean;
  isLoadingMetrics: boolean;
  metricCount?: number;
  metrics: Array<Metric>;
  servicesMetrics?: ListingModel<ServiceMetric>;
}

const isResourcesString = (resources: Array<SelectEntry> | string) =>
  equals(type(resources), 'String');

export const useListMetrics = ({
  resources,
  selectedMetrics = []
}: Props): UseListMetricsState => {
  const resourcesWithString = resources.filter((resource) =>
    isResourcesString(resource.resources)
  );
  const { data: servicesMetrics, isFetching: isLoadingMetrics } = useFetchQuery<
    ListingModel<ServiceMetric>
  >({
    decoder: serviceMetricsDecoder,
    getEndpoint: () =>
      buildListingEndpoint({
        baseEndpoint: metricsEndpoint,
        parameters: {
          limit: 1000,
          search: {
            conditions: isEmpty(resourcesWithString)
              ? undefined
              : resourcesWithString.map((resource) => ({
                  field: buildResourceTypeNameForSearchParameter(
                    resource.resourceType
                  ),
                  values: {
                    $rg: resource.resources
                  }
                })),
            lists: resources
              .filter((resource) => !isResourcesString(resource.resources))
              .map((resource) => ({
                field: resourceTypeQueryParameter[resource.resourceType],
                values: equals(resource.resourceType, 'service')
                  ? pluck('name', resource.resources)
                  : pluck('id', resource.resources)
              }))
          }
        }
      }),
    getQueryKey: () => ['metrics', JSON.stringify(resources)],
    queryOptions: {
      enabled:
        !isEmpty(resources) &&
        all((resource) => !isEmpty(resource.resources), resources),
      suspense: false
    }
  });

  const hasTooManyMetrics = gt(servicesMetrics?.meta?.total || 0, 1000);

  const metricCount = servicesMetrics?.meta?.total;

  const unitsFromSelectedMetrics = pipe(
    flatten,
    pluck('unit'),
    uniq
  )(selectedMetrics || []);

  const hasMultipleUnitsSelected = gt(length(unitsFromSelectedMetrics), 1);

  const metrics: Array<Metric> = pipe(
    pluck('metrics'),
    flatten,
    uniqBy(({ name }) => name)
  )(servicesMetrics?.result || []);

  return {
    hasMultipleUnitsSelected,
    hasTooManyMetrics,
    isLoadingMetrics,
    metricCount,
    metrics,
    servicesMetrics
  };
};
