<?php

/*
 * Copyright 2005 - 2025 Centreon (https://www.centreon.com/)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * For more information : contact@centreon.com
 *
 */

namespace CentreonLegacy\Core\Module;

use Centreon\Test\Mock\DependencyInjector\ServiceContainer;
use CentreonLegacy\Core\Configuration\Configuration;
use CentreonLegacy\ServiceProvider;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Pimple\Psr11\Container;
use VirtualFileSystem\FileSystem;

/**
 * @group CentreonLegacy
 * @group CentreonLegacy\Module
 */
class HealthcheckTest extends TestCase
{
    /** @var FileSystem */
    public $fileSystem;

    /** @var MockObject&ServiceContainer */
    public $container;

    /** @var MockObject&Healthcheck */
    public $service;

    public function setUp(): void
    {
        // mount VFS
        $this->fileSystem = new FileSystem();

        $this->fileSystem->createDirectory('/tmp');
        $this->fileSystem->createDirectory('/tmp/checklist');
        $this->fileSystem->createFile('/tmp/checklist/requirements.php', '');

        $this->fileSystem->createDirectory('/tmp1');
        $this->fileSystem->createDirectory('/tmp1/checklist');

        $this->container = new ServiceContainer();
        $this->container[ServiceProvider::CONFIGURATION] = $this
            ->getMockBuilder(Configuration::class)
            ->disableOriginalConstructor()
            ->onlyMethods([
                'getModulePath',
            ])
            ->getMock();

        $this->container[ServiceProvider::CONFIGURATION]
            ->method('getModulePath')
            ->will(
                $this->returnCallback(function () {
                    return $this->fileSystem->path('/');
                })
            );

        $this->service = $this->getMockBuilder(Healthcheck::class)
            ->setConstructorArgs([
                new Container($this->container),
            ])
            ->onlyMethods([
                'getRequirements',
            ])
            ->getMock();

        $this->setRequirementMockMethodValue();
    }

    public function tearDown(): void
    {
        $this->container->terminate();
        $this->container = null;
    }

    public function testCheckWithDotModuleName(): void
    {
        try {
            $this->service->check('.');
        } catch (\Exception $ex) {
            $this->assertInstanceOf(Exception\HealthcheckNotFoundException::class, $ex);
        }
    }

    public function testCheckWithMissingModule(): void
    {
        try {
            $this->service->check('mod');
        } catch (\Exception $ex) {
            $this->assertInstanceOf(Exception\HealthcheckNotFoundException::class, $ex);
        }
    }

    public function testCheckWithWarning(): void
    {
        $module = 'tmp';

        $this->setRequirementMockMethodValue(null, null, true);

        try {
            $this->service->check($module);
        } catch (\Exception $ex) {
            $this->assertInstanceOf(Exception\HealthcheckWarningException::class, $ex);
        }
    }

    public function testCheckWithCritical(): void
    {
        $module = 'tmp';
        $valueMessages = [
            [
                'ErrorMessage' => 'err',
                'Solution' => 'none',
            ],
        ];

        $this->setRequirementMockMethodValue($valueMessages, null, false, true);

        try {
            $this->service->check($module);
        } catch (\Exception $ex) {
            $this->assertInstanceOf(Exception\HealthcheckCriticalException::class, $ex);
            $this->assertEquals($valueMessages[0], $this->service->getMessages());
        }
    }

    public function testCheckWithoutRequirementsFile(): void
    {
        $module = 'tmp1';

        $this->setRequirementMockMethodValue();

        try {
            $this->service->check($module);
        } catch (\Exception $ex) {
            $this->assertInstanceOf(Exception\HealthcheckNotFoundException::class, $ex);
        }
    }

    public function testCheck(): void
    {
        $module = 'tmp';
        $valueTime = time();
        $valueCustomAction = [
            'action' => 'act',
            'name' => 'nm',
        ];

        $this->setRequirementMockMethodValue(null, $valueCustomAction, false, false, $valueTime);

        $result = $this->service->check($module);
        $this->assertTrue($result);
        $this->assertEquals($valueTime, $this->service->getLicenseExpiration()->getTimestamp());
        $this->assertEquals(
            [
                'customAction' => $valueCustomAction['action'],
                'customActionName' => $valueCustomAction['name'],
            ],
            $this->service->getCustomAction()
        );
    }

    public function testCheckPrepareResponseWithNotFound(): void
    {
        $module = 'mod';
        $value = [
            'status' => 'notfound',
        ];

        $result = $this->service->checkPrepareResponse($module);

        $this->assertEquals($result, $value);
    }

    public function testCheckPrepareResponseWithCritical(): void
    {
        $module = 'tmp';
        $valueMessages = [
            [
                'ErrorMessage' => 'err',
                'Solution' => 'none',
            ],
        ];
        $value = [
            'status' => 'critical',
            'message' => [
                'ErrorMessage' => $valueMessages[0]['ErrorMessage'],
                'Solution' => $valueMessages[0]['Solution'],
            ],
        ];

        $this->setRequirementMockMethodValue($valueMessages, null, false, true);

        $result = $this->service->checkPrepareResponse($module);

        $this->assertEquals($result, $value);
    }

    public function testCheckPrepareResponseWithWarning(): void
    {
        $module = 'tmp';
        $valueMessages = [
            [
                'ErrorMessage' => 'err',
                'Solution' => 'none',
            ],
        ];
        $value = [
            'status' => 'warning',
            'message' => [
                'ErrorMessage' => $valueMessages[0]['ErrorMessage'],
                'Solution' => $valueMessages[0]['Solution'],
            ],
        ];

        $this->setRequirementMockMethodValue($valueMessages, null, true);

        $result = $this->service->checkPrepareResponse($module);

        $this->assertEquals($value, $result);
    }

    public function testCheckPrepareResponseWithException(): void
    {
        $module = 'tmp';
        $valueException = 'test exception';
        $value = [
            'status' => 'critical',
            'message' => [
                'ErrorMessage' => $valueException,
                'Solution' => '',
            ],
        ];

        $this->service
            ->method('getRequirements')
            ->will($this->returnCallback(
                function (
                    $checklistDir,
                    &$message,
                    &$customAction,
                    &$warning,
                    &$critical,
                    &$licenseExpiration,
                ) use ($valueException): void {
                    throw new \Exception($valueException);
                }
            ));

        $result = $this->service->checkPrepareResponse($module);

        $this->assertEquals($result, $value);
    }

    public function testCheckPrepareResponse(): void
    {
        $module = 'tmp';
        $valueTime = time();
        $valueCustomAction = [
            'action' => 'act',
            'name' => 'nm',
        ];
        $value = [
            'status' => 'ok',
            'customAction' => $valueCustomAction['action'],
            'customActionName' => $valueCustomAction['name'],
            'licenseExpiration' => $valueTime,
        ];

        $this->setRequirementMockMethodValue(null, $valueCustomAction, false, false, $valueTime);

        $result = $this->service->checkPrepareResponse($module);

        $this->assertEquals($result, $value);
    }

    public function testReset(): void
    {
        $this->service->reset();
        $this->assertEquals(null, $this->service->getMessages());
        $this->assertEquals(null, $this->service->getCustomAction());
        $this->assertEquals(null, $this->service->getLicenseExpiration());
    }

    /**
     * Set up method getRequirements
     *
     * @param array $messageV
     * @param array $customActionV
     * @param bool $warningV
     * @param bool $criticalV
     * @param int $licenseExpirationV
     */
    protected function setRequirementMockMethodValue(
        $messageV = null,
        $customActionV = null,
        $warningV = false,
        $criticalV = false,
        $licenseExpirationV = null,
    ) {
        $this->service
            ->method('getRequirements')
            ->will($this->returnCallback(
                function (
                    $checklistDir,
                    &$message,
                    &$customAction,
                    &$warning,
                    &$critical,
                    &$licenseExpiration,
                ) use ($messageV, $customActionV, $warningV, $criticalV, $licenseExpirationV): void {
                    $message = $messageV ?: [];
                    $customAction = $customActionV;
                    $warning = $warningV;
                    $critical = $criticalV;
                    $licenseExpiration = $licenseExpirationV;
                }
            ));
    }
}
