<?php

/*
 * Copyright 2005 - 2025 Centreon (https://www.centreon.com/)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * For more information : contact@centreon.com
 *
 */

declare(strict_types=1);

namespace Tests\Core\Domain\Configuration\User\Model;

use Centreon\Domain\Common\Assertion\AssertionException;
use Core\Domain\Configuration\User\Model\User;
use PHPUnit\Framework\TestCase;

class UserTest extends TestCase
{
    /**
     * Test user creation with empty alias
     */
    public function testAliasEmpty(): void
    {
        $alias = '';

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage(
            AssertionException::minLength(
                $alias,
                mb_strlen($alias),
                User::MIN_ALIAS_LENGTH,
                'User::alias'
            )->getMessage()
        );

        new User(
            1,
            $alias,
            'name',
            'email',
            false,
            'light',
            'compact',
            true
        );
    }

    /**
     * Test user creation with too long alias
     */
    public function testAliasTooLong(): void
    {
        $alias = str_repeat('a', 256);

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage(
            AssertionException::maxLength(
                $alias,
                mb_strlen($alias),
                User::MAX_ALIAS_LENGTH,
                'User::alias'
            )->getMessage()
        );

        new User(
            1,
            $alias,
            'name',
            'email',
            false,
            'light',
            'compact',
            true
        );
    }

    /**
     * Test user creation with empty name
     */
    public function testNameEmpty(): void
    {
        $name = '';

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage(
            AssertionException::minLength(
                $name,
                mb_strlen($name),
                User::MIN_NAME_LENGTH,
                'User::name'
            )->getMessage()
        );

        new User(
            1,
            'alias',
            $name,
            'email',
            false,
            'light',
            'compact',
            true
        );
    }

    /**
     * Test user creation with too long name
     */
    public function testNameTooLong(): void
    {
        $name = str_repeat('a', 256);

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage(
            AssertionException::maxLength(
                $name,
                mb_strlen($name),
                User::MAX_NAME_LENGTH,
                'User::name'
            )->getMessage()
        );

        new User(
            1,
            'alias',
            $name,
            'email',
            false,
            'light',
            'compact',
            true
        );
    }

    /**
     * Test user creation with empty email
     */
    public function testEmailEmpty(): void
    {
        $email = '';

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage(
            AssertionException::minLength(
                $email,
                mb_strlen($email),
                User::MIN_EMAIL_LENGTH,
                'User::email'
            )->getMessage()
        );

        new User(
            1,
            'alias',
            'name',
            $email,
            false,
            'light',
            'compact',
            true
        );
    }

    /**
     * Test user creation with too long email
     */
    public function testEmailTooLong(): void
    {
        $email = str_repeat('a', 256) . '@centreon.com';

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage(
            AssertionException::maxLength(
                $email,
                mb_strlen($email),
                User::MAX_EMAIL_LENGTH,
                'User::email'
            )->getMessage()
        );

        new User(
            1,
            'alias',
            'name',
            $email,
            false,
            'light',
            'compact',
            true
        );
    }

    /**
     * Test user creation with too long userInterfaceViewMode
     */
    public function testUserInterfaceViewModeTooLong(): void
    {
        $userInterfaceViewMode = str_repeat('a', 256);

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage(
            AssertionException::maxLength(
                $userInterfaceViewMode,
                mb_strlen($userInterfaceViewMode),
                User::MAX_USER_INTERFACE_DENSITY_LENGTH,
                'User::userInterfaceViewMode'
            )->getMessage()
        );

        new User(
            1,
            'alias',
            'name',
            'hellotest@centreon.com',
            false,
            'light',
            $userInterfaceViewMode,
            true
        );
    }

    /**
     * Test user creation with empty userInterfaceViewMode
     */
    public function testUserInterfaceViewModeEmpty(): void
    {
        $userInterfaceViewMode = '';

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage(
            AssertionException::notEmptyString(
                'User::userInterfaceViewMode'
            )->getMessage()
        );

        new User(
            1,
            'alias',
            'name',
            'hellotest@centreon.com',
            false,
            'light',
            $userInterfaceViewMode,
            true
        );
    }

    /**
     * Test user creation with userInterfaceViewMode not handled
     */
    public function testUserInterfaceViewModeNotHandled(): void
    {
        $userInterfaceViewMode = 'random-ui';

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('User interface view mode provided not handled');

        new User(
            1,
            'alias',
            'name',
            'hellotest@centreon.com',
            false,
            'light',
            $userInterfaceViewMode,
            true
        );
    }

    /**
     * Test user creation
     */
    public function testUserCreation(): void
    {
        $id = 1;
        $alias = 'alias';
        $name = 'name';
        $email = 'root@localhost';
        $isAdmin = true;
        $theme = 'light';
        $userInterfaceViewMode = 'compact';

        $user = new User(
            $id,
            $alias,
            $name,
            $email,
            $isAdmin,
            'light',
            'compact',
            true
        );

        $this->assertEquals($id, $user->getId());
        $this->assertEquals($alias, $user->getAlias());
        $this->assertEquals($name, $user->getName());
        $this->assertEquals($email, $user->getEmail());
        $this->assertEquals($isAdmin, $user->isAdmin());
        $this->assertEquals($theme, $user->getTheme());
        $this->assertEquals($userInterfaceViewMode, $user->getUserInterfaceDensity());
    }
}
