<?php

/*
 * Copyright 2005 - 2025 Centreon (https://www.centreon.com/)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * For more information : contact@centreon.com
 *
 */

/**
 * Class
 *
 * @class CentreonEscalation
 */
class CentreonEscalation
{
    /** @var CentreonDB */
    protected $db;

    /**
     * CentreonEscalation constructor
     *
     * @param CentreonDB $pearDB
     */
    public function __construct($pearDB)
    {
        $this->db = $pearDB;
    }

    /**
     * @param int $field
     * @return array
     */
    public static function getDefaultValuesParameters($field)
    {
        $parameters = [];
        $parameters['currentObject']['table'] = 'escalation';
        $parameters['currentObject']['id'] = 'esc_id';
        $parameters['currentObject']['name'] = 'esc_name';
        $parameters['currentObject']['comparator'] = 'esc_id';

        switch ($field) {
            case 'esc_cgs':
                $parameters['type'] = 'relation';
                $parameters['externalObject']['object'] = 'centreonContactgroup';
                $parameters['relationObject']['table'] = 'escalation_contactgroup_relation';
                $parameters['relationObject']['field'] = 'contactgroup_cg_id';
                $parameters['relationObject']['comparator'] = 'escalation_esc_id';
                break;
            case 'esc_hServices':
                $parameters['type'] = 'relation';
                $parameters['externalObject']['object'] = 'centreonService';
                $parameters['relationObject']['table'] = 'escalation_service_relation';
                $parameters['relationObject']['field'] = 'host_host_id';
                $parameters['relationObject']['additionalField'] = 'service_service_id';
                $parameters['relationObject']['comparator'] = 'escalation_esc_id';
                break;
            case 'esc_hosts':
                $parameters['type'] = 'relation';
                $parameters['externalObject']['object'] = 'centreonHost';
                $parameters['relationObject']['table'] = 'escalation_host_relation';
                $parameters['relationObject']['field'] = 'host_host_id';
                $parameters['relationObject']['comparator'] = 'escalation_esc_id';
                break;
            case 'esc_hgs':
                $parameters['type'] = 'relation';
                $parameters['externalObject']['object'] = 'centreonHostgroups';
                $parameters['relationObject']['table'] = 'escalation_hostgroup_relation';
                $parameters['relationObject']['field'] = 'hostgroup_hg_id';
                $parameters['relationObject']['comparator'] = 'escalation_esc_id';
                break;
            case 'esc_sgs':
                $parameters['type'] = 'relation';
                $parameters['externalObject']['object'] = 'centreonServicegroups';
                $parameters['relationObject']['table'] = 'escalation_servicegroup_relation';
                $parameters['relationObject']['field'] = 'servicegroup_sg_id';
                $parameters['relationObject']['comparator'] = 'escalation_esc_id';
                break;
            case 'esc_metas':
                $parameters['type'] = 'relation';
                $parameters['externalObject']['object'] = 'centreonMeta';
                $parameters['relationObject']['table'] = 'escalation_meta_service_relation';
                $parameters['relationObject']['field'] = 'meta_service_meta_id';
                $parameters['relationObject']['comparator'] = 'escalation_esc_id';
                break;
        }

        return $parameters;
    }

    /**
     * @param array $values
     * @param array $options
     *
     * @throws PDOException
     * @return array
     */
    public function getObjectForSelect2($values = [], $options = [])
    {
        global $centreon;
        $items = [];

        // get list of authorized host categories
        if (! $centreon->user->access->admin) {
            $hcAcl = $centreon->user->access->getHostCategories();
        }

        $listValues = '';
        $queryValues = [];
        if (! empty($values)) {
            foreach ($values as $k => $v) {
                $listValues .= ':hc' . $v . ',';
                $queryValues['hc' . $v] = (int) $v;
            }
            $listValues = rtrim($listValues, ',');
        } else {
            $listValues .= '""';
        }

        // get list of selected host categories
        $query = 'SELECT hc_id, hc_name FROM hostcategories '
            . 'WHERE hc_id IN (' . $listValues . ') ORDER BY hc_name ';

        $stmt = $this->db->prepare($query);

        if ($queryValues !== []) {
            foreach ($queryValues as $key => $id) {
                $stmt->bindValue(':' . $key, $id, PDO::PARAM_INT);
            }
        }
        $stmt->execute();

        while ($row = $stmt->fetchRow()) {
            // hide unauthorized host categories
            $hide = false;
            if (! $centreon->user->access->admin && count($hcAcl) && ! in_array($row['hc_id'], array_keys($hcAcl))) {
                $hide = true;
            }

            $items[] = ['id' => $row['hc_id'], 'text' => $row['hc_name'], 'hide' => $hide];
        }

        return $items;
    }
}
