<?php

/*
 * Copyright 2005 - 2025 Centreon (https://www.centreon.com/)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * For more information : contact@centreon.com
 *
 */

ini_set('display_errors', 'Off');

require_once realpath(__DIR__ . '/../../../../../../bootstrap.php');
include_once _CENTREON_PATH_ . 'www/class/centreonUtils.class.php';
include_once _CENTREON_PATH_ . 'www/class/centreonXMLBGRequest.class.php';
include_once _CENTREON_PATH_ . 'www/include/monitoring/status/Common/common-Func.php';
include_once _CENTREON_PATH_ . 'www/include/common/common-Func.php';

// Create XML Request Objects
CentreonSession::start(1);
$obj = new CentreonXMLBGRequest($dependencyInjector, session_id(), 1, 1, 0, 1);

if (! isset($obj->session_id) || ! CentreonSession::checkSession($obj->session_id, $obj->DB)) {
    echo 'Bad Session ID';

    exit();
}

// Set Default Poller
$obj->getDefaultFilters();

$centreon = $_SESSION['centreon'];

/**
 * true: URIs will correspond to deprecated pages
 * false: URIs will correspond to new page (Resource Status)
 */
$useDeprecatedPages = $centreon->user->doesShowDeprecatedPages();

// Check Arguments From GET tab
$o = isset($_GET['o']) ? HtmlAnalyzer::sanitizeAndRemoveTags($_GET['o']) : 'h';
$p = filter_input(INPUT_GET, 'p', FILTER_VALIDATE_INT, ['options' => ['default' => 2]]);
$num = filter_input(INPUT_GET, 'num', FILTER_VALIDATE_INT, ['options' => ['default' => 0]]);
$limit = filter_input(INPUT_GET, 'limit', FILTER_VALIDATE_INT, ['options' => ['default' => 20]]);
// if instance value is not set, displaying all active pollers linked resources
$instance = filter_var($obj->defaultPoller ?? -1, FILTER_VALIDATE_INT);
$hostgroups = filter_var($obj->defaultHostgroups ?? 0, FILTER_VALIDATE_INT);
$search = isset($_GET['search']) ? HtmlAnalyzer::sanitizeAndRemoveTags($_GET['search']) : '';
$sortType = isset($_GET['sort_type']) ? HtmlAnalyzer::sanitizeAndRemoveTags($_GET['sort_type']) : 'host_name';
$order = isset($_GET['order']) && $_GET['order'] === 'DESC' ? 'DESC' : 'ASC';

// Backup poller selection
$obj->setInstanceHistory($instance);

$kernel = App\Kernel::createForWeb();
$resourceController = $kernel->getContainer()->get(
    Centreon\Application\Controller\MonitoringResourceController::class
);

$service = [];
$host_status = [];
$service_status = [];
$host_services = [];
$metaService_status = [];
$tab_host_service = [];
$tabIcone = [];
// saving bound values
$queryValues = [];

/**
 * Get status
 */
$rq1 = 'SELECT SQL_CALC_FOUND_ROWS DISTINCT 1 AS REALTIME, hosts.name, hosts.state, hosts.icon_image, hosts.host_id FROM hosts ';
if ($hostgroups) {
    $rq1 .= ', hosts_hostgroups hg, hostgroups hg2 ';
}

if (! $obj->is_admin) {
    $rq1 .= ', centreon_acl ';
}

$rq1 .= "WHERE hosts.name NOT LIKE '\_Module\_%' AND hosts.enabled = 1 "
    . $obj->access->queryBuilder('AND', 'hosts.host_id', 'centreon_acl.host_id') . ' '
    . $obj->access->queryBuilder('AND', 'group_id', $obj->grouplistStr) . ' ';

if (str_ends_with($o, '_pb') || str_ends_with($o, '_ack_0')) {
    $rq1 .= 'AND hosts.host_id IN ( '
        . 'SELECT s.host_id FROM services s '
        . 'WHERE s.state != 0 '
        . 'AND s.state != 4 '
        . 'AND s.enabled = 1) ';
} elseif (str_ends_with($o, '_ack_1')) {
    $rq1 .= 'AND hosts.host_id IN ( '
        . 'SELECT s.host_id FROM services s '
        . "WHERE s.acknowledged = '1' "
        . 'AND s.enabled = 1) ';
}

if ($search != '') {
    $rq1 .= 'AND hosts.name like :search ';
    $queryValues['search'] = [PDO::PARAM_STR => '%' . $search . '%'];
}

if ($instance != -1) {
    $rq1 .= 'AND hosts.instance_id = :instance ';
    $queryValues['instance'] = [PDO::PARAM_INT => $instance];
}

if ($hostgroups) {
    $rq1 .= ' AND hosts.host_id = hg.host_id
        AND hg.hostgroup_id = :hostGroup
        AND hg.hostgroup_id = hg2.hostgroup_id ';
    $queryValues['hostGroup'] = [PDO::PARAM_INT => $hostgroups];
}

// Sort order
switch ($sortType) {
    case 'current_state':
        $rq1 .= 'ORDER BY hosts.state ' . $order . ',hosts.name ';
        break;
    default:
        $rq1 .= 'ORDER BY hosts.name ' . $order . ' ';
        break;
}

// Limit
$rq1 .= ' LIMIT :numLimit, :limit';
$queryValues['numLimit'] = [PDO::PARAM_INT => ($num * $limit)];
$queryValues['limit'] = [PDO::PARAM_INT => $limit];

$dbResult = $obj->DBC->prepare($rq1);
foreach ($queryValues as $bindId => $bindData) {
    foreach ($bindData as $bindType => $bindValue) {
        $dbResult->bindValue($bindId, $bindValue, $bindType);
    }
}
$dbResult->execute();

$numRows = (int) $obj->DBC->query('SELECT FOUND_ROWS() AS REALTIME')->fetchColumn();

// Info / Pagination
$obj->XML->startElement('reponse');
$obj->XML->startElement('i');
$obj->XML->writeElement('numrows', $numRows);
$obj->XML->writeElement('num', $num);
$obj->XML->writeElement('limit', $limit);
$obj->XML->writeElement('p', $p);
$obj->XML->endElement();

$buildParameter = function (string $id, string $name) {
    return [
        'id' => $id,
        'name' => $name,
    ];
};

$buildServicesUri = function (string $hostname, array $statuses) use ($resourceController, $buildParameter) {
    return $resourceController->buildListingUri([
        'filter' => json_encode([
            'criterias' => [
                'search' => 'h.name:^' . $hostname . '$',
                'resourceTypes' => [$buildParameter('service', 'Service')],
                'statuses' => $statuses,
            ],
        ]),
    ]);
};

$okStatus = $buildParameter('OK', 'Ok');
$warningStatus = $buildParameter('WARNING', 'Warning');
$criticalStatus = $buildParameter('CRITICAL', 'Critical');
$unknownStatus = $buildParameter('UNKNOWN', 'Unknown');
$pendingStatus = $buildParameter('PENDING', 'Pending');

$ct = 0;
$tabFinal = [];
while ($ndo = $dbResult->fetch()) {
    $tabFinal[$ndo['name']]['nb_service_k'] = 0;
    $tabFinal[$ndo['name']]['host_id'] = $ndo['host_id'];
    if (! str_starts_with($o, 'svcSum')) {
        $tabFinal[$ndo['name']]['nb_service_k'] = $obj->monObj->getServiceStatusCount(
            $ndo['name'],
            $obj,
            $o,
            CentreonMonitoring::SERVICE_STATUS_OK
        );
    }
    $tabFinal[$ndo['name']]['nb_service_w'] = 0 + $obj->monObj->getServiceStatusCount(
        $ndo['name'],
        $obj,
        $o,
        CentreonMonitoring::SERVICE_STATUS_WARNING
    );
    $tabFinal[$ndo['name']]['nb_service_c'] = 0 + $obj->monObj->getServiceStatusCount(
        $ndo['name'],
        $obj,
        $o,
        CentreonMonitoring::SERVICE_STATUS_CRITICAL
    );
    $tabFinal[$ndo['name']]['nb_service_u'] = 0 + $obj->monObj->getServiceStatusCount(
        $ndo['name'],
        $obj,
        $o,
        CentreonMonitoring::SERVICE_STATUS_UNKNOWN
    );
    $tabFinal[$ndo['name']]['nb_service_p'] = 0 + $obj->monObj->getServiceStatusCount(
        $ndo['name'],
        $obj,
        $o,
        CentreonMonitoring::SERVICE_STATUS_PENDING
    );
    $tabFinal[$ndo['name']]['cs'] = $ndo['state'];

    $tabIcone[$ndo['name']] = isset($ndo['icon_image']) && $ndo['icon_image'] != '' ? $ndo['icon_image'] : 'none';
}

foreach ($tabFinal as $host_name => $tab) {
    $obj->XML->startElement('l');
    $obj->XML->writeAttribute('class', $obj->getNextLineClass());
    $obj->XML->writeElement('o', $ct++);
    $obj->XML->writeElement('hn', CentreonUtils::escapeSecure($host_name), false);
    $obj->XML->writeElement('hnl', CentreonUtils::escapeSecure(urlencode($host_name)));
    $obj->XML->writeElement('hid', $tab['host_id'], false);
    $obj->XML->writeElement(
        'h_details_uri',
        $useDeprecatedPages
            ? 'main.php?p=20202&o=hd&host_name=' . $host_name
            : $resourceController->buildHostDetailsUri($tab['host_id'])
    );
    $serviceListingDeprecatedUri = 'main.php?p=20201&o=svc&host_search=' . $host_name;
    $obj->XML->writeElement(
        's_listing_uri',
        $useDeprecatedPages
            ? 'main.php?o=svc&p=20201&statusFilter=&host_search=' . $host_name
            : $resourceController->buildListingUri([
                'filter' => json_encode([
                    'criterias' => [
                        'search' => 'h.name:^' . $host_name . '$',
                    ],
                ]),
            ])
    );
    $obj->XML->writeElement('ico', $tabIcone[$host_name]);
    $obj->XML->writeElement('hs', _($obj->statusHost[$tab['cs']]), false);
    $obj->XML->writeElement('hc', $obj->colorHost[$tab['cs']]);
    $obj->XML->writeElement('sc', $tab['nb_service_c']);
    $obj->XML->writeElement('scc', $obj->colorService[2]);
    $obj->XML->writeElement('sw', $tab['nb_service_w']);
    $obj->XML->writeElement('swc', $obj->colorService[1]);
    $obj->XML->writeElement('su', $tab['nb_service_u']);
    $obj->XML->writeElement('suc', $obj->colorService[3]);
    $obj->XML->writeElement('sk', $tab['nb_service_k']);
    $obj->XML->writeElement('skc', $obj->colorService[0]);
    $obj->XML->writeElement('sp', $tab['nb_service_p']);
    $obj->XML->writeElement('spc', $obj->colorService[4]);
    $obj->XML->writeElement(
        's_listing_ok',
        $useDeprecatedPages
            ? $serviceListingDeprecatedUri . '&statusFilter=ok'
            : $buildServicesUri($host_name, [$okStatus])
    );
    $obj->XML->writeElement(
        's_listing_warning',
        $useDeprecatedPages
            ? $serviceListingDeprecatedUri . '&statusFilter=warning'
            : $buildServicesUri($host_name, [$warningStatus])
    );
    $obj->XML->writeElement(
        's_listing_critical',
        $useDeprecatedPages
            ? $serviceListingDeprecatedUri . '&statusFilter=critical'
            : $buildServicesUri($host_name, [$criticalStatus])
    );
    $obj->XML->writeElement(
        's_listing_unknown',
        $useDeprecatedPages
            ? $serviceListingDeprecatedUri . '&statusFilter=unknown'
            : $buildServicesUri($host_name, [$unknownStatus])
    );
    $obj->XML->writeElement(
        's_listing_pending',
        $useDeprecatedPages
            ? $serviceListingDeprecatedUri . '&statusFilter=pending'
            : $buildServicesUri($host_name, [$pendingStatus])
    );
    $obj->XML->writeElement('chartIcon', returnSvg('www/img/icons/chart.svg', 'var(--icons-fill-color)', 18, 18));
    $obj->XML->writeElement('viewIcon', returnSvg('www/img/icons/view.svg', 'var(--icons-fill-color)', 18, 18));
    $obj->XML->endElement();
}

if (! $ct) {
    $obj->XML->writeElement('infos', 'none');
}
$obj->XML->endElement();

// Send Header
$obj->header();

// Send XML
$obj->XML->output();
