import { Given, Then, When } from '@badeball/cypress-cucumber-preprocessor';

import { PAGES } from 'fixtures/shared/constants/pages';
import data from '../../../fixtures/acls/acl-data.json';

const originalAclGroup = {
  alias: 'ACL group',
  linkedContactGroups: data.contactGroups.contactGroup1.name,
  name: 'ACL_group'
};

const modifiedAclGroup = {
  alias: 'ACL group modified',
  name: 'ACL_group_modified'
};

const duplicatedAclGroup = {
  name: `${originalAclGroup.name}_1`
};

beforeEach(() => {
  cy.startContainers();
  cy.intercept({
    method: 'GET',
    url: '/centreon/api/internal.php?object=centreon_topology&action=navigationList'
  }).as('getNavigationList');
  cy.intercept({
    method: 'GET',
    url: '/centreon/api/internal.php?object=centreon_topcounter&action=user'
  }).as('getTopCounter');
  cy.intercept({
    method: 'GET',
    url: '/centreon/include/common/userTimezone.php'
  }).as('getTimeZone');
  cy.intercept(
    'HEAD',
    'https://guide.centreon.com/agent/static/b06b875d-4a10-4365-7edf-8efeaf53dfdd/pendo.js'
  ).as('pendoRequest');
});

afterEach(() => {
  cy.stopContainers();
});

Given('I am logged in a Centreon server', () => {
  cy.loginByTypeOfUser({ jsonName: 'admin' });
});

Given('one contact group exists including two non admin contacts', () => {
  cy.addContact({
    admin: data.contacts.contact1.admin,
    email: data.contacts.contact1.email,
    name: data.contacts.contact1.name,
    password: data.contacts.contact1.password
  });
  cy.addContact({
    admin: data.contacts.contact2.admin,
    email: data.contacts.contact2.email,
    name: data.contacts.contact2.name,
    password: data.contacts.contact2.password
  });
  cy.addContactGroup({
    contacts: [data.contacts.contact1.name, data.contacts.contact2.name],
    name: data.contactGroups.contactGroup1.name
  });
});

When('the access group is saved with its properties', () => {
  cy.visit(PAGES.configuration.aclAccessGroupsLegacy);
  cy.wait('@getTimeZone');
  cy.getIframeBody().contains('a', 'Add').click();

  cy.wait('@getTimeZone');
  cy.getIframeBody()
    .find('input[name="acl_group_name"]')
    .type(originalAclGroup.name);
  cy.getIframeBody()
    .find('input[name="acl_group_alias"]')
    .type(originalAclGroup.alias);
  cy.getIframeBody()
    .find('select[name="cg_contactGroups-f[]"]')
    .select(originalAclGroup.linkedContactGroups);

  cy.getIframeBody().find('input[name="add"]').eq(1).click();

  cy.getIframeBody()
    .find('select[name="cg_contactGroups-t[]"]')
    .should('contain', originalAclGroup.linkedContactGroups);

  cy.getIframeBody().find('input[name="submitA"]').eq(0).click();
});

When('a menu access is linked with this group', () => {
  cy.addACLMenu({ name: data.ACLMenu.name, rule: ['Home'] });
  cy.addAclMenuToAclGroup({
    aclGroupName: originalAclGroup.name,
    aclMenuName: data.ACLMenu.name
  });
});

Then(
  'all linked users have the access list group displayed in Centreon authentication tab',
  () => {
    cy.logout();
    cy.loginByCredentials({
      login: data.contacts.contact1.name,
      password: data.contacts.contact1.password
    });

    cy.wait('@getTopCounter');
    cy.getByTestId({ tag: 'button', testId: 'Mini Centreon Logo' }).click();
    cy.contains('span', 'Home').should('exist');

    cy.logout();
    cy.loginByCredentials({
      login: data.contacts.contact2.name,
      password: data.contacts.contact2.password
    });

    cy.wait('@getTopCounter');
    cy.getByTestId({ tag: 'button', testId: 'Mini Centreon Logo' }).click();
    cy.contains('span', 'Home').should('exist');
  }
);

Given('a new access group with a linked contact group', () => {
  cy.addContact({
    admin: data.contacts.contact1.admin,
    email: data.contacts.contact1.email,
    name: data.contacts.contact1.name,
    password: data.contacts.contact1.password
  });
  cy.addContact({
    admin: data.contacts.contact2.admin,
    email: data.contacts.contact2.email,
    name: data.contacts.contact2.name,
    password: data.contacts.contact2.password
  });
  cy.addContactGroup({
    contacts: [data.contacts.contact1.name, data.contacts.contact2.name],
    name: data.contactGroups.contactGroup1.name
  });
});

Then(
  'the contact group has the access group displayed in Relations information',
  () => {
    cy.visit(PAGES.configuration.contactGroupsLegacy);

    cy.wait(['@getTimeZone', '@pendoRequest']).then(() => {
      cy.executeActionOnIframe(
        data.contactGroups.contactGroup1.name,
        ($body) => {
          cy.wrap($body)
            .contains(data.contactGroups.contactGroup1.name)
            .eq(0)
            .click();
        },
        3,
        3000
      );
    });

    cy.wait(['@getTimeZone', '@pendoRequest']).then(() => {
      cy.executeActionOnIframe(
        originalAclGroup.name,
        ($body) => {
          cy.wrap($body)
            .find('select[name="cg_acl_groups[]"]')
            .contains(originalAclGroup.name);
        },
        3,
        3000
      );
    });
  }
);

Given('one existing ACL access group', () => {
  cy.addACLGroup({ name: originalAclGroup.name });
});

When('I modify its properties', () => {
  cy.visit(PAGES.configuration.aclAccessGroupsLegacy);
  cy.wait('@getTimeZone');

  cy.getIframeBody().contains(originalAclGroup.name).click();

  cy.wait('@getTimeZone');

  cy.getIframeBody()
    .find('input[name="acl_group_name"]')
    .type(modifiedAclGroup.name);
  cy.getIframeBody()
    .find('input[name="acl_group_alias"]')
    .type(modifiedAclGroup.alias);

  cy.getIframeBody().find('input[name="submitC"]').eq(1).click();
});

Then('all modified properties are updated', () => {
  cy.wait('@getTimeZone');

  cy.getIframeBody().should('contain', modifiedAclGroup.name);
  cy.getIframeBody().should('contain', modifiedAclGroup.alias);
});

When('I duplicate the access group', () => {
  cy.visit(PAGES.configuration.aclAccessGroupsLegacy);
  cy.wait('@getTimeZone');

  cy.getIframeBody()
    .contains('tr', originalAclGroup.name)
    .within(() => {
      cy.get('td.ListColPicker').click();
    });

  cy.get<HTMLIFrameElement>('iframe#main-content', { timeout: 10000 }).then(
    (iframe: JQuery<HTMLIFrameElement>) => {
      const win = iframe[0].contentWindow;

      if (!win) {
        throw new Error('Cannot get iframe');
      }

      cy.stub(win, 'confirm').returns(true);
    }
  );

  cy.getIframeBody().find('select[name="o1"]').select('Duplicate');
});

Then('a new access group with identical properties is created', () => {
  cy.wait('@getTimeZone');

  const originalAclGroupValues: Array<string> = [];
  cy.getIframeBody()
    .contains('tr', originalAclGroup.name)
    .within(() => {
      cy.get('td').each((td, index) => {
        if (index >= 2 && index <= 5)
          originalAclGroupValues.push(td.text().trim());
      });
    });

  const duplicatedAclGroupValues: Array<string> = [];
  cy.getIframeBody()
    .contains('tr', duplicatedAclGroup.name)
    .within(() => {
      cy.get('td').each((td, index) => {
        if (index >= 2 && index <= 5)
          duplicatedAclGroupValues.push(td.text().trim());
      });
    });

  cy.wrap(duplicatedAclGroupValues).then((duplicatedValues) => {
    for (let i = 0; i < originalAclGroupValues.length; i += 1) {
      expect(duplicatedValues[i]).to.equal(originalAclGroupValues[i]);
    }
  });
});

When('I delete the access group', () => {
  cy.visit(PAGES.configuration.aclAccessGroupsLegacy);
  cy.wait('@getTimeZone');

  cy.getIframeBody()
    .contains('tr', originalAclGroup.name)
    .within(() => {
      cy.get('td.ListColPicker').click();
    });

  cy.get<HTMLIFrameElement>('iframe#main-content', { timeout: 10000 }).then(
    (iframe: JQuery<HTMLIFrameElement>) => {
      const win = iframe[0].contentWindow;

      if (!win) {
        throw new Error('Cannot get iframe');
      }

      cy.stub(win, 'confirm').returns(true);
    }
  );

  cy.getIframeBody().find('select[name="o1"]').select('Delete');
});

Then('it does not exist anymore', () => {
  cy.wait('@getTimeZone');

  cy.getIframeBody().should('not.contain', originalAclGroup.name);
});

Given('one existing enabled ACL access group', () => {
  cy.addACLGroup({ name: originalAclGroup.name });
});

When('I disable it', () => {
  cy.visit(PAGES.configuration.aclAccessGroupsLegacy);
  cy.wait('@getTimeZone');

  cy.getIframeBody().contains(originalAclGroup.name).click();

  cy.wait('@getTimeZone');

  cy.getIframeBody()
    .find('input[name="acl_group_activate[acl_group_activate]"][value="0"]')
    .parent()
    .click();

  cy.getIframeBody().find('input[name="submitC"]').eq(1).click();
});

Then('its status is modified', () => {
  cy.wait('@getTimeZone');

  cy.getIframeBody()
    .contains('tr', originalAclGroup.name)
    .should('contain', 'Disabled');
});
