import { PAGES } from 'fixtures/shared/constants/pages';
import metrics from '../../fixtures/dashboards/creation/widgets/metrics.json';

interface Dashboard {
  description?: string;
  name: string;
}

interface HostDataType {
  acknowledgementTimeout: number;
  actionUrl: string;
  activeCheckEnabled: number;
  addInheritedContact: boolean;
  addInheritedContactGroup: boolean;
  address: string;
  alias: string;
  categories: Array<number>;
  checkCommandArgs: Array<string>;
  checkCommandId: number;
  checkTimePeriodId: number;
  comment: string;
  eventHandlerCommandArgs: Array<string>;
  eventHandlerCommandId: number;
  eventHandlerEnabled: number;
  firstNotificationDelay: number;
  flapDetectionEnabled: number;
  freshnessChecked: number;
  freshnessThreshold: number;
  geoCoords: string;
  groups: Array<number>;
  highFlapThreshold: number;
  iconAlternative: string;
  iconId: number;
  isActivated: boolean;
  lowFlapThreshold: number;
  macros: Array<object>;
  maxCheckAttempts: number;
  monitoringServerId: number;
  name: string;
  normalCheckInterval: number;
  note: string;
  noteUrl: string;
  notificationEnabled: number;
  notificationInterval: number;
  notificationOptions: number;
  notificationTimePeriodId: number;
  passiveCheckEnabled: number;
  recoveryNotificationDelay: number;
  retryCheckInterval: number;
  severityId: number;
  snmpCommunity: string;
  snmpVersion: string;
  templates: Array<number>;
  timezoneId: number;
}

interface ServiceDataType {
  acknowledgementTimeout: number;
  actionUrl: string;
  activeCheckEnabled: number;
  checkCommandArgs: Array<string>;
  checkCommandId: number | null;
  checkTimePeriodId: number | null;
  comment: string;
  eventHandlerCommandArgs: Array<string>;
  eventHandlerCommandId: number;
  eventHandlerEnabled: number;
  firstNotificationDelay: number;
  flapDetectionEnabled: number;
  freshnessChecked: number;
  freshnessThreshold: number;
  notificationEnabled: number;
  isContactAdditiveInheritance: boolean;
  isContactGroupAdditiveInheritance: boolean;
  notificationInterval: number;
  notificationTimePeriodId: number;
  notificationType: number;
  retryCheckInterval: number;
  recoveryNotificationDelay: number;
  lowFlapThreshold: number;
  passiveCheckEnabled: number;
  macros: Array<object>;
  geoCoords: string;
  name: string;
  highFlapThreshold: number;
  normalCheckInterval: number;
  maxCheckAttempts: number;
  serviceTemplateId: number | null;
  graphTemplateId: number | null;
  note: string;
  noteUrl: string;
  hostId: number;
  iconId: number | null;
  iconAlternative: string;
  severityId: number;
  isActivated: boolean;
  serviceCategories: Array<number>;
  serviceGroups: Array<number>;
  volatilityEnabled: number;
}

Cypress.Commands.add('enableDashboardFeature', () => {
  cy.execInContainer({
    command: `sed -i 's@"dashboard": [0-3]@"dashboard": 3@' /usr/share/centreon/config/features.json`,
    name: 'web'
  });
});

Cypress.Commands.add('visitDashboards', () => {
  cy.intercept({
    method: 'GET',
    times: 1,
    url: '/centreon/api/latest/configuration/dashboards*'
  }).as('listAllDashboards');

  const dashboardsUrl = PAGES.monitoring.dashboardsLibrary;
  cy.url().then((url) =>
    url.includes(dashboardsUrl)
      ? cy.visit(dashboardsUrl)
      : cy.visit(PAGES.monitoring.dashboards)
  );

  cy.wait('@listAllDashboards');
});

Cypress.Commands.add('visitDashboard', (name) => {
  cy.visitDashboards();
  cy.contains(name).click();

  cy.url().should('match', /\/home\/dashboards\/library\/\d+$/);
});

Cypress.Commands.add('editDashboard', (name) => {
  cy.visitDashboard(name);

  cy.getByLabel({
    label: 'Edit dashboard',
    tag: 'button'
  }).click();

  cy.url().should('match', /\/home\/dashboards\/library\/\d+\?edit=true/);

  cy.getByLabel({
    label: 'Save',
    tag: 'button'
  }).should('be.visible');
});

Cypress.Commands.add('editWidget', (nameOrPosition) => {
  if (typeof nameOrPosition === 'string') {
    cy.contains('div.react-grid-item', nameOrPosition).as('widgetItem');
  } else {
    cy.get('div.react-grid-item')
      .eq(nameOrPosition - 1)
      .as('widgetItem');
  }

  cy.get('@widgetItem').within(() => {
    cy.getByTestId({ testId: 'More actions' }).should('be.visible').click();
  });

  cy.getByLabel({
    label: 'Edit widget',
    tag: 'li'
  })
    .should('exist') // do not check with "be.visible" because it can be hidden by the tooltip of "more actions" button
    .click();

  cy.contains('Widget properties').should('be.visible');
});

Cypress.Commands.add(
  'waitUntilForDashboardRoles',
  (accessRightsTestId: string, expectedElementCount: number) => {
    const openModalAndCheck = (): Cypress.Chainable<boolean> => {
      cy.getByTestId({ testId: accessRightsTestId }).invoke('show').click();
      cy.get('.MuiSelect-select').should('be.visible');

      return cy
        .get('.MuiSelect-select')
        .should('be.visible')
        .then(($elements) => {
          cy.getByLabel({ label: 'close', tag: 'button' }).click();
          return cy.wrap($elements.length === expectedElementCount);
        });
    };

    return cy.waitUntil(() => openModalAndCheck(), {
      errorMsg: 'The expected number of elements was not found',
      interval: 3000,
      timeout: 30000
    });
  }
);

Cypress.Commands.add('verifyGraphContainer', () => {
  cy.get('[class*="graphContainer"]')
    .should('be.visible')
    .within(() => {
      cy.get('[class*="graphText"]')
        .should('be.visible')
        .within(() => {
          cy.get('[class*="MuiTypography-h2"]').should('be.visible');

          cy.get('[class*="MuiTypography-h5"]')
            .eq(0)
            .should('contain', metrics.rtaValues.warning);

          cy.get('[class*="MuiTypography-h5"]')
            .eq(1)
            .should('contain', metrics.rtaValues.critical);
        });
    });
});

Cypress.Commands.add('verifyDuplicatesGraphContainer', () => {
  cy.get('[class*="graphContainer"]')
    .eq(1)
    .should('be.visible')
    .within(() => {
      cy.get('[class*="graphText"]')
        .should('be.visible')
        .within(() => {
          cy.get('[class*="MuiTypography-h2"]').should('be.visible');

          cy.get('[class*="MuiTypography-h5"]')
            .eq(0)
            .should('contain', metrics.plValues.warning);

          cy.get('[class*="MuiTypography-h5"]')
            .eq(1)
            .should('contain', metrics.plValues.critical);
        });
    });
});

Cypress.Commands.add('waitUntilPingExists', () => {
  cy.intercept({
    method: 'GET',
    url: /\/centreon\/api\/latest\/monitoring\/services\/names.*$/
  }).as('servicesRequest');

  return cy.waitUntil(
    () => {
      cy.getByTestId({ testId: 'Select resource' }).eq(0).realClick();
      cy.contains('Centreon-Server').realClick();
      cy.getByTestId({ testId: 'Select resource' }).eq(0).blur();
      cy.getByTestId({ testId: 'Select resource' }).eq(1).realClick();

      return cy.wait('@servicesRequest').then((interception) => {
        if (interception?.response) {
          cy.log('Response Body:', interception.response.body);
          const responseBody = interception.response.body;
          if (
            Array.isArray(responseBody.result) &&
            responseBody.result.length > 0
          ) {
            const pingFound = responseBody.result.some(
              (result) => result.name === 'Ping'
            );

            if (pingFound) {
              cy.contains('Ping').click();
              return cy.wrap(true);
            }
            cy.log('Ping not found in the API response');

            return cy.wrap(false);
          }
          cy.log('Response is not an array or is empty');
          return cy.wrap(false);
        }
        cy.log('Interception or response is undefined');
        return cy.wrap(false);
      });
    },
    {
      errorMsg: 'Timed out waiting for Ping to exist',
      interval: 3000,
      timeout: 60000
    }
  );
});

Cypress.Commands.add(
  'insertDashboardWithWidget',
  (dashboardBody, patchBody, widgetName, widgetType) => {
    cy.request({
      body: { ...dashboardBody },
      method: 'POST',
      url: '/centreon/api/latest/configuration/dashboards'
    }).then((response) => {
      const dashboardId = response.body.id;

      cy.waitUntil(
        () => {
          return cy
            .request({
              method: 'GET',
              url: `/centreon/api/latest/configuration/dashboards/${dashboardId}`
            })
            .then((getResponse) => {
              return getResponse.body && getResponse.body.id === dashboardId;
            });
        },
        { timeout: 10000 }
      );

      const formData = new FormData();

      formData.append('panels[0][name]', widgetName);
      formData.append('panels[0][widget_type]', widgetType);

      formData.append('panels[0][layout][x]', '0');
      formData.append('panels[0][layout][y]', '0');
      formData.append('panels[0][layout][width]', '12');
      formData.append('panels[0][layout][height]', '3');
      formData.append('panels[0][layout][min_width]', '2');
      formData.append('panels[0][layout][min_height]', '2');

      formData.append('panels[0][widget_settings]', JSON.stringify(patchBody));

      cy.request({
        body: formData,
        headers: {
          'Content-Type': 'multipart/form-data'
        },
        method: 'POST',
        url: `/centreon/api/latest/configuration/dashboards/${dashboardId}`
      }).then((patchResponse) => {
        cy.log('Widget added successfully:', patchResponse);
      });
    });
  }
);

Cypress.Commands.add(
  'insertDashboardWithDoubleWidget',
  (dashboardBody, patchBody1, patchBody2, widgetName, widgetType) => {
    cy.request({
      body: { ...dashboardBody },
      method: 'POST',
      url: '/centreon/api/latest/configuration/dashboards'
    }).then((response) => {
      const dashboardId = response.body.id;

      cy.waitUntil(
        () => {
          return cy
            .request({
              method: 'GET',
              url: `/centreon/api/latest/configuration/dashboards/${dashboardId}`
            })
            .then((getResponse) => {
              return getResponse.body && getResponse.body.id === dashboardId;
            });
        },
        { timeout: 10000 }
      );

      const formData = new FormData();

      // Panel 1
      formData.append('panels[0][name]', widgetName);
      formData.append('panels[0][widget_type]', widgetType);
      formData.append('panels[0][layout][x]', '0');
      formData.append('panels[0][layout][y]', '0');
      formData.append('panels[0][layout][width]', '12');
      formData.append('panels[0][layout][height]', '3');
      formData.append('panels[0][layout][min_width]', '2');
      formData.append('panels[0][layout][min_height]', '2');
      formData.append('panels[0][widget_settings]', JSON.stringify(patchBody1));

      // Panel 2
      formData.append('panels[1][name]', widgetName);
      formData.append('panels[1][widget_type]', widgetType);
      formData.append('panels[1][layout][x]', '0');
      formData.append('panels[1][layout][y]', '3');
      formData.append('panels[1][layout][width]', '12');
      formData.append('panels[1][layout][height]', '3');
      formData.append('panels[1][layout][min_width]', '2');
      formData.append('panels[1][layout][min_height]', '2');
      formData.append('panels[1][widget_settings]', JSON.stringify(patchBody2));

      // Log form data
      const dataToLog = {};
      formData.forEach((value, key) => {
        dataToLog[key] = value;
      });

      cy.log('FormData before POST:', JSON.stringify(dataToLog, null, 2));

      cy.request({
        body: formData,
        headers: {
          'Content-Type': 'multipart/form-data'
        },
        method: 'POST',
        url: `/centreon/api/latest/configuration/dashboards/${dashboardId}`
      }).then((patchResponse) => {
        cy.log('Widget added successfully:', patchResponse);
      });
    });
  }
);

Cypress.Commands.add('getCellContent', (rowIndex, columnIndex) => {
  cy.waitUntil(
    () =>
      cy
        .get(
          `.MuiTable-root:eq(1) .MuiTableRow-root:nth-child(${rowIndex}) .MuiTableCell-root:nth-child(${columnIndex})`
        )
        .should('be.visible')
        .then(() => true),
    { interval: 1000, timeout: 10000 }
  );

  return cy
    .get(
      `.MuiTable-root:eq(1) .MuiTableRow-root:nth-child(${rowIndex}) .MuiTableCell-root:nth-child(${columnIndex})`
    )
    .invoke('text')
    .then((content) => {
      const columnContents = content ? content.match(/[A-Z][a-z]*/g) || [] : [];
      cy.log(
        `Column contents (${rowIndex}, ${columnIndex}): ${columnContents
          .join(',')
          .trim()}`
      );

      return cy.wrap(columnContents);
    });
});

Cypress.Commands.add('applyAcl', () => {
  const apacheUser = Cypress.env('WEB_IMAGE_OS').includes('alma')
    ? 'apache'
    : 'www-data';

  cy.execInContainer({
    command: `su -s /bin/sh ${apacheUser} -c "/usr/bin/env php -q /usr/share/centreon/cron/centAcl.php"`,
    name: 'web'
  });
});

Cypress.Commands.add(
  'verifyLegendItemStyle',
  (
    index: number,
    expectedColors: Array<string>,
    expectedValues: Array<string>,
    alternativeValues: Array<string> = []
  ): Cypress.Chainable => {
    cy.get('[data-testid="Legend"] > *')
      .eq(index)
      .each(($legendItem) => {
        cy.wrap($legendItem)
          .find('[class*=legendItem] a')
          .then(($aTags) => {
            $aTags.each((i, aTag) => {
              cy.wrap(aTag)
                .find('div')
                .invoke('attr', 'style')
                .then((style) => {
                  expect(style).to.contain(expectedColors[i]);
                });

              cy.wrap(aTag)
                .next('p')
                .invoke('text')
                .then((text) => {
                  const possibleValues = [expectedValues[i]];

                  if (alternativeValues[i]) {
                    possibleValues.push(alternativeValues[i]);
                  }

                  expect(text.trim()).to.match(
                    new RegExp(possibleValues.join('|'))
                  );
                });
            });
          });
      });

    return cy;
  }
);

Cypress.Commands.add('addNewHostAndReturnId', (hostData = {}) => {
  const defaultHostData = {
    address: '127.0.0.1',
    alias: 'generic-active-host',
    groups: [53],
    macros: [
      {
        description: 'Some text to describe the macro',
        // biome-ignore lint/style/useNamingConvention: <explanation>
        is_password: false,
        name: 'MacroName',
        value: 'macroValue'
      }
    ],
    // biome-ignore lint/style/useNamingConvention: <explanation>
    monitoring_server_id: 1,
    name: 'generic-active-host',
    templates: [2]
  };

  const requestBody = { ...defaultHostData, ...hostData };

  cy.request({
    body: requestBody,
    method: 'POST',
    url: '/centreon/api/latest/configuration/hosts'
  }).then((response) => {
    expect(response.status).to.eq(201);
    return response.body.id;
  });
});

Cypress.Commands.add('getServiceIdByName', (serviceName) => {
  return cy
    .request({
      method: 'GET',
      url: '/centreon/api/latest/monitoring/services'
    })
    .then((response) => {
      const service = response.body.result.find(
        (s) => s.display_name === serviceName
      );
      if (service) {
        return service.id;
      }
      throw new Error(`Service with name ${serviceName} not found`);
    });
});

Cypress.Commands.add('patchServiceWithHost', (hostId, serviceId) => {
  const patchData = {
    // biome-ignore lint/style/useNamingConvention: <explanation>
    host_id: hostId
  };
  cy.request({
    body: patchData,
    method: 'PATCH',
    url: `/centreon/api/latest/configuration/services/${serviceId}`
  }).then((response) => {
    expect(response.status).to.eq(204);
  });
});

Cypress.Commands.add(
  'addNewServiceAndReturnId',
  (hostId: number, serviceData = {}) => {
    const defaultServiceData = {
      // biome-ignore lint/style/useNamingConvention: <explanation>
      check_command_args: [],
      // biome-ignore lint/style/useNamingConvention: <explanation>
      check_command_id: null,
      comment: 'string',
      // biome-ignore lint/style/useNamingConvention: <explanation>
      geo_coords: '48.10,12.5',
      // biome-ignore lint/style/useNamingConvention: <explanation>
      host_id: hostId,
      // biome-ignore lint/style/useNamingConvention: <explanation>
      max_check_attempts: 1,
      name: 'generic-service',
      // biome-ignore lint/style/useNamingConvention: <explanation>
      service_template_id: 5
    };

    const requestBody = {
      ...defaultServiceData,
      ...serviceData,
      // biome-ignore lint/style/useNamingConvention: <explanation>
      host_id: hostId
    };

    cy.request({
      body: requestBody,
      method: 'POST',
      url: '/centreon/api/latest/configuration/services'
    }).then((response) => {
      expect(response.status).to.eq(201);
      return response.body.id;
    });
  }
);

Cypress.Commands.add(
  'addMultipleHosts',
  (
    numberOfHosts = 20
  ): Cypress.Chainable<{
    hostIds: Array<number>;
    serviceIds: Array<number>;
  }> => {
    const hostIds: Array<number> = [];
    const serviceIds: Array<number> = [];

    let chain = cy.wrap(null);

    for (let i = 0; i < numberOfHosts; i++) {
      const uniqueHostData = {
        alias: `generic-active-host-alias-${i + 1}`,
        name: `generic-active-host-${i + 1}`
      };

      const uniqueServiceData = {
        // biome-ignore lint/style/useNamingConvention: <explanation>
        geo_coords: '48.10,12.5',
        name: `service-${i + 1}`
      };

      chain = chain.then(() => {
        return cy
          .addNewHostAndReturnId(uniqueHostData)
          .then((hostId: number) => {
            hostIds.push(hostId);

            return cy
              .addNewServiceAndReturnId(hostId, uniqueServiceData)
              .then((serviceId: number | null) => {
                if (serviceId !== null) {
                  serviceIds.push(serviceId);
                }
                return cy.wrap(null);
              });
          });
      });
    }

    return chain.then(() => {
      cy.log('All hosts and services have been created and associated.');
      return cy.wrap({ hostIds, serviceIds });
    });
  }
);

Cypress.Commands.add(
  'waitForElementToBeVisible',
  (selector, timeout = 50000, interval = 2000) => {
    cy.waitUntil(
      () =>
        cy.get('body').then(($body) => {
          const element = $body.find(selector);

          return element.length > 0 && element.is(':visible');
        }),
      {
        errorMsg: `The element '${selector}' is not visible`,
        interval,
        timeout
      }
    ).then((isVisible) => {
      if (!isVisible) {
        throw new Error(`The element '${selector}' is not visible`);
      }
    });
  }
);

declare global {
  // biome-ignore lint/style/noNamespace: <explanation>
  namespace Cypress {
    interface Chainable {
      addMultipleHosts(
        numberOfHosts?: number
      ): Chainable<{ hostIds: Array<number>; serviceIds: Array<number> }>;
      addNewHostAndReturnId: (
        hostData?: Partial<HostDataType>
      ) => Cypress.Chainable;
      addNewServiceAndReturnId(
        hostId: number,
        serviceData?: Partial<ServiceDataType>
      ): Chainable<number>;
      applyAcl: () => Cypress.Chainable;
      editDashboard: (name: string) => Cypress.Chainable;
      editWidget: (nameOrPosition: string | number) => Cypress.Chainable;
      enableDashboardFeature: () => Cypress.Chainable;
      getCellContent: (rowIndex: number, colIndex: number) => Cypress.Chainable;
      getServiceIdByName: (serviceName: string) => Cypress.Chainable;
      insertDashboardWithDoubleWidget: (
        dashboard: Dashboard,
        // biome-ignore lint/suspicious/noExplicitAny: <explanation>
        patchBody1: Record<string, any>,
        // biome-ignore lint/suspicious/noExplicitAny: <explanation>
        patchBody2: Record<string, any>,
        widgetName: string,
        widgetType: string
      ) => Cypress.Chainable;
      patchServiceWithHost: (
        hostId: string,
        serviceId: string
      ) => Cypress.Chainable;
      insertDashboardWithWidget: (
        dashboard: Dashboard,
        // biome-ignore lint/suspicious/noExplicitAny: <explanation>
        patchBody: Record<string, any>,
        widgetName: string,
        widgetType: string
        // biome-ignore lint/suspicious/noExplicitAny: <explanation>
      ) => Chainable<any>;
      verifyDuplicatesGraphContainer: (metrics) => Cypress.Chainable;
      verifyGraphContainer: (metrics) => Cypress.Chainable;
      verifyLegendItemStyle: (
        index: number,
        expectedColors: Array<string>,
        expectedValue: Array<string>,
        alternativeValues?: Array<string>
      ) => Cypress.Chainable;
      visitDashboard: (name: string) => Cypress.Chainable;
      visitDashboards: () => Cypress.Chainable;
      waitForElementToBeVisible(
        selector: string,
        timeout?: number,
        interval?: number
      ): Cypress.Chainable;
      waitUntilForDashboardRoles: (
        accessRightsTestId: string,
        expectedElementCount: number
      ) => Cypress.Chainable<boolean>;
      waitUntilPingExists: () => Cypress.Chainable;
    }
  }
}
