/*
  Copyright (c) 2016-2024, Smart Engines Service LLC
  All rights reserved.
*/

/**
 * @file id_face_session.h
 * @brief id.engine face session declaration
 */

#ifndef IDENGINE_ID_FACE_SESSION_H_INCLUDED
#define IDENGINE_ID_FACE_SESSION_H_INCLUDED

#include <se_common_common.h>
#include <id_face_result_idengine.h>

namespace se { namespace id {

/**
 * @brief The main processing class for the face matching and analysis
 *        functionality of Smart ID Engine
 */
class SE_DLL_EXPORT IdFaceSession {
public:
  /// Default dtor
  virtual ~IdFaceSession() = default;

  virtual const char* GetActivationRequest() = 0;

  virtual void Activate(const char* activation_response) = 0;

  virtual bool IsActivated() const = 0;

  /**
   * @brief Analyzes a face in the input image (independent from session state)
   * @param face_image - input image to analyze
   * @return An IdFaceDescription object. The object is allocated, the caller
   *         is responsible for deleting it.
   */
  virtual IdFaceDescription* GetFaceDescription(
      const se::common::Image& face_image) const = 0;

  /**
   * @brief Returns the similarity result for the two provided face images
   *        (independent from session state)
   * @param face_image_a - lvalue image for comparison
   * @param face_image_b - rvalue image for comparison
   * @return A similarity comparison result object
   */
  virtual IdFaceSimilarityResult GetSimilarity(
      const se::common::Image& face_image_a,
      const se::common::Image& face_image_b) const = 0;

  /**
   * @brief Returns the similarity result for the two provided face descriptions
   *        (independent from session state)
   * @param face_description_a - lvalue face description for comparison
   * @param face_description_b - rvalue face description for comparison
   * @return A similarity comparison result object
   */
  virtual IdFaceSimilarityResult GetSimilarity(
      const IdFaceDescription& face_description_a,
      const IdFaceDescription& face_description_b) const = 0;

  /**
   * @brief Adds a new face image to the current session object
   * @param face_image - the image of a face to be added
   */
  virtual void AddFaceImage(const se::common::Image& face_image) = 0;

  /**
   * @brief Adds a new face image (by description) to the current session.
   *        This method is an alternative to AddFaceImage() if the caller
   *        already have described the face image.
   * @param face_description - the description of a face to be added
   */
  virtual void AddFaceDescription(const IdFaceDescription& face_description) = 0;

  /**
   * @brief Checks whether the session has an accumulated face description
   * @return Returns true iff the session has an accumulated face description
   */
  virtual bool HasAccumulatedFaceDescription() const = 0;

  /**
   * @brief Returns face description accumulated in the session
   * @return An IdFaceDescription (const ref) for the accumulated description
   */
  virtual const IdFaceDescription& GetAccumulatedFaceDescription() const = 0;

  /**
   * @brief Returns the similarity result for the stream of images stored in the
   *        session state (lvalue) with an passed rvalue image
   * @param compare_image - the rvalue image to compare the state with
   * @return A similarity comparison result object
   */
  virtual IdFaceSimilarityResult GetSimilarityWith(
      const se::common::Image& compare_image) const = 0;

  /**
   * @brief Returns the similarity result for the stream of images stored in the
   *        session state (lvalue) with an passed rvalue face description
   * @param compare_description - rvalue description to compare the state with
   * @return A similarity comparison result object
   */
  virtual IdFaceSimilarityResult GetSimilarityWith(
      const IdFaceDescription& compare_description) const = 0;

  /**
   * @brief Returns the liveness estimation result for the stream of images
   *        passed through the session
   * @return A liveness estimation result object
   */
  virtual IdFaceLivenessResult GetLivenessResult() const = 0;

  /**
   * @brief Resets the session state
   */
  virtual void Reset() = 0;
};


} } // namespace se::id

#endif // IDENGINE_ID_FACE_SESSION_H_INCLUDED
