/*
  Copyright (c) 2016-2024, Smart Engines Service LLC
  All rights reserved.
*/

/**
 * @file id_face_result.h
 * @brief id.engine face results declaration
 */

#ifndef IDENGINE_ID_FACE_RESULT_H_INCLUDED
#define IDENGINE_ID_FACE_RESULT_H_INCLUDED

#include <se_common_common.h>

namespace se { namespace id {

/**
 * @brief The enumeration to encode possible face image orientations
 */
enum SE_DLL_EXPORT IdFaceOrientation {
  IdFaceOrientation_0,    ///< Upright orientation
  IdFaceOrientation_90CW, ///< Rotated 90 degrees clockwise
  IdFaceOrientation_180,  ///< Rotated 180 degrees
  IdFaceOrientation_270CW ///< Rotated 270 degrees clockwise
};

enum SE_DLL_EXPORT IdFaceStatus {
  IdFaceStatus_NotUsed,            ///< Was created but not used
  IdFaceStatus_Success,            ///< Everything alright
  IdFaceStatus_A_FaceNotFound,     ///< Face was not found for image A
  IdFaceStatus_B_FaceNotFound,     ///< Face was not found for image B
  IdFaceStatus_FaceNotFound,       ///< There is no face found
  IdFaceStatus_NoAccumulatedResult ///< Face matching with session where is no Accumulated result
};

/// Forward-declaration of the internal IdFaceDescription implementation
class IdFaceDescriptionImpl;

class SE_DLL_EXPORT IdFaceDescription {
public:

  /**
   * @brief Factory method for creating an empty face description
   * @return Pointer to a created description. New object is allocated, the
   *         caller is responsible for deleting it.
   */
  static IdFaceDescription* CreateEmpty();

public:
  /// Non-trivial dtor
  ~IdFaceDescription();

  /**
   * @brief Clones a face description
   * @return  Pointer to a cloned description. New object is allocated, the
   *          caller is responsible for deleting it.
   */
  IdFaceDescription* Clone() const;

public:
  /// Returns true iff the face was detected
  bool FaceDetected() const;

  /// Gets a normalized face rectangle
  const se::common::Rectangle& GetRectangle() const;

  /// Sets a normalized face rectangle
  void SetRectangle(const se::common::Rectangle& rect);

  /// Gets face orientation
  IdFaceOrientation GetOrientation() const;

  /// Sets face orientation
  void SetOrientation(IdFaceOrientation orientation);

  /// Returns true iff the result has a visualization image
  bool HasVisualizationImage() const;

  /// Gets the visualization image (const ref)
  const se::common::Image& GetVisualizationImage() const;

  /// Gets the visualization image (mutable ref)
  se::common::Image& GetMutableVisualizationImage();

  /// Sets the visualization image
  void SetVisualizationImage(const se::common::Image& image);

  /// Gets size of the feature vector
  int GetFeatureVectorSize() const;

  /// Gets feature from the feature vector
  float GetFeature(int index) const;

  /// Sets feature in the feature vector by index
  void SetFeature(int index, float value);

  /// Resizees the feature vector to provided size
  void ResizeFeatureVector(int size);

public:
  /// Returns internal implementation (const ref)
  const IdFaceDescriptionImpl& GetImpl() const;

  /// Returns internal implementation (mutable ref)
  IdFaceDescriptionImpl& GetMutableImpl();

  /// Returns new object from implementation
  static IdFaceDescription* ConstructFromImpl(
      IdFaceDescriptionImpl* impl);

private:
  /// Private constructor from implementation
  IdFaceDescription(IdFaceDescriptionImpl* impl);

  IdFaceDescriptionImpl* pimpl_; ///< internal implementation
};

/// Forward-declaration of the internal IdFaceLivenessResult implementation
class IdFaceLivenessResultImpl;

/**
 * @brief The class which represents the face liveness result
 */
class SE_DLL_EXPORT IdFaceLivenessResult {
public:
  /// Non-trivial dtor
  ~IdFaceLivenessResult();

  /// Main ctor - stores the liveness estimation value
  IdFaceLivenessResult(double liveness_estimation = 0.0);

  /// Copy ctor
  IdFaceLivenessResult(const IdFaceLivenessResult& copy);

  /// Assignment operator
  IdFaceLivenessResult& operator =(const IdFaceLivenessResult& other);

public:
  /// Returns the liveness estimation value (double in range [0.0, 1.0])
  double GetLivenessEstimation() const;

  /// Sets the liveness estimation value
  void SetLivenessEstimation(double liveness_estimation);

  /// Returns pointer to the start of the instruction char*
  const char* GetLivenessInstruction() const;

  /// Sets instruction to check liveness
  void SetLivenessInstruction(const char* instruction);

private:
  IdFaceLivenessResultImpl* pimpl_; ///< internal implementation
};

/// Forward-declaration of the IdFaceSimilarityResult internal implementation
class IdFaceSimilarityResultImpl;

/**
 * @brief The class representing the face similarity comparison result
 */
class SE_DLL_EXPORT IdFaceSimilarityResult {
public:
  /// Non-trivial dtor
  ~IdFaceSimilarityResult();

  /// Main ctor - stores the similarity estimation value
  IdFaceSimilarityResult(double similarity_estimation = 0.0, IdFaceStatus status = IdFaceStatus_NotUsed);

  /// Copy ctor
  IdFaceSimilarityResult(const IdFaceSimilarityResult& copy);

  /// Assignment operator
  IdFaceSimilarityResult& operator = (const IdFaceSimilarityResult& other);

public:
  /// Gets the faces similarity estimation value (dobule in range [0.0, 1.0])
  double GetSimilarityEstimation() const;

  /// Sets the faces similarity estimation value
  void SetSimilarityEstimation(double similarity_estimation);

  /// Get the process status
  IdFaceStatus GetStatus() const;

  /// Set the process status
  void SetStatus(const IdFaceStatus& status);

private:
  IdFaceSimilarityResultImpl* pimpl_; ///< internal implementation
};


} } // namespace se::id

#endif // IDENGINE_ID_FACE_RESULT_H_INCLUDED
