/*
  Copyright (c) 2016-2022, Smart Engines Service LLC
  All rights reserved.
*/

/**
 * @file se_bytestring.h
 * @brief secommon library byte string implementation
 */

#ifndef SECOMMON_SE_BYTE_STRING_H_INCLUDED
#define SECOMMON_SE_BYTE_STRING_H_INCLUDED

#include <se_export_defs_common.h>
#include <se_serialization_common.h>
#include <stdint.h>
#include <string>

namespace se {
namespace common {

class SE_DLL_EXPORT ByteString
{
public:
  /// Default ctor, creates an empty string
  ByteString();

  /// Ctor from a C-string
  explicit ByteString(const char* c_str);

  /// Copy ctor
  ByteString(const ByteString& other);

  /// Assignment operator
  ByteString& operator=(const ByteString& other);

  /// Non-trivial dtor
  ~ByteString();

  /// Appends a string to this instance
  ByteString& operator+=(const ByteString& other);

  /// Creates a concatenation of this instance and the other string
  ByteString operator+(const ByteString& other) const;

  /// Provides const-access to the byte contained in the given position.
  const uint8_t& at(size_t pos) const;

  /// Returns the length of the bytestring.
  size_t size() const;

  /// Returns true if bytestring is empty.
  bool empty() const;

  /// Serializes the string given a serializer object
  void Serialize(Serializer& serializer) const;

  /// Internal serialization implementation
  void SerializeImpl(SerializerImplBase& serializer_impl) const;

private:
  /// Safety check used only from at().
  void _range_check(size_t pos) const;

private:
  size_t len_;   ///< length of the internal buffer in bytes
  uint8_t* buf_; ///< internal buffer
};

/**
 * @brief Converts bytestring into hex-value string.
 * @param bytes - input bytes.
 * @return hex-value string.
 */
inline std::string
to_hex_string(const ByteString& bytes)
{
  static const int npos = 3;
  std::string hexstr(bytes.size() * npos, ' ');
  for (size_t i = 0; i < bytes.size(); ++i)
    ::sprintf(&hexstr[i * npos], "%02X ", bytes.at(i));

  return hexstr.substr(0, hexstr.size() - 1);
}

} // namespace common
} // namespace se

#endif // SECOMMON_SE_BYTE_STRING_H_INCLUDED
