/*
  Copyright (c) 2016-2024, Smart Engines Service LLC
  All rights reserved.
*/

#import <Foundation/Foundation.h>
#import <AVFoundation/AVFoundation.h>
#import <UIKit/UIKit.h>

#import "id_result.h"
#import "id_engine.h"

#import "se_image.h"

@protocol SmartIDEngineDelegate <NSObject>
@optional
- (void) SmartIDEngineObtainedResult:(nonnull SEIdResult *)result;
- (void) SmartIDEngineObtainedDetectionResult:(nonnull SEIdTemplateDetectionResult *)result;
- (void) SmartIDEngineObtainedSegmentationResult:(nonnull SEIdTemplateSegmentationResult *)result;
- (void) SmartIDEngineObtainedFeedback:(nonnull SEIdFeedbackContainer *)feedback;

- (void) SmartIDEngineObtainedResult:(nonnull SEIdResult *)result
                 fromFrameWithBuffer:(nonnull CMSampleBufferRef)buffer;

- (void) SmartIDEngineObtainedSingleImageResult:(nonnull SEIdResult *)result;
@end

@protocol SmartIDEngineInitializationDelegate <NSObject>
@optional
- (void) SmartIDEngineInitialized;
- (void) SmartIDEngineVideoSessionStarted;
- (void) SmartIDEngineVideoSessionDismissed;
@end

@protocol SmartLivenessDelegate <NSObject>
@optional
- (void) SmartLivenessObtainedMessage:(nonnull NSString *)message;
- (void) SmartLivenessObtainedResult:(nonnull SEIdFaceLivenessResult *)result;
@end

@interface SmartIDEngineInstance : NSObject

@property (weak, nonatomic, nullable, readonly) id<SmartIDEngineDelegate> engineDelegate;
@property (weak, nonatomic, nullable, readonly) id<SmartLivenessDelegate> livenessDelegate;
@property (weak, nonatomic, nullable, readonly) id<SmartIDEngineInitializationDelegate> initializationDelegate;

@property (strong, nullable, readonly) SEIdEngine* engine; // main configuration of Smart ID Engine
@property (strong, nullable, readonly) SEIdSession* videoSession; // current video recognition session
@property (strong, nullable, readonly) SEIdFaceSession* livenessSession; // current liveness recognition session
@property (strong, nullable) SEIdSessionSettings* sessionSettings; // current session settings
@property (strong, nullable) SEIdFaceSessionSettings* faceSessionSettings; // current face session settings

// Best frame image finder section
@property (nonatomic, assign, readonly) BOOL isBestImageFrameEnabled;
@property (nonnull,readonly) NSString* bestImageFrame; // current image frame
@property (nonnull,readonly) NSMutableDictionary* frameImageTemplatesInfo;

- (nonnull NSString*) bestImageFrame;
- (void) findBestImageFrame;
- (nonnull NSString*) getBestImageFrame;

@property BOOL engineInitialized;
@property BOOL videoSessionRunning;

- (nonnull instancetype) initWithSignature:(nonnull NSString *)signature;
- (void) setEngineDelegate:(nullable __weak id<SmartIDEngineDelegate>)delegate;
- (void) setLivenessDelegate:(nullable __weak id<SmartLivenessDelegate>)delegate;
- (void) setInitializationDelegate:(nullable __weak id<SmartIDEngineInitializationDelegate>)delegate;

- (void) initializeEngine:(nonnull NSString *)bundlePath;

- (BOOL) initVideoSession;



- (void) initLivenessSession;

- (void) dismissVideoSession;
- (void) dismissLivenessSession;
- (void) dismissVideoSessionRunning;

- (void) processFrame:(nonnull CMSampleBufferRef)sampleBuffer
      withOrientation:(UIDeviceOrientation)deviceOrientation;

- (void) processFrame:(nonnull CMSampleBufferRef)sampleBuffer
      withOrientation:(UIDeviceOrientation)deviceOrientation
               andRoi:(CGRect)roi;

- (void) processLivenessFrame:(nonnull CMSampleBufferRef)sampleBuffer
      withOrientation:(UIDeviceOrientation)deviceOrientation;

- (nonnull SEIdResult*) processSingleImage:(nonnull SECommonImageRef *)image;
- (nonnull SEIdResult*) processSingleImageFromFile:(nonnull NSString *)filePath;
- (nonnull SEIdResult*) processSingleImageFromUIImage:(nonnull UIImage *)image;

- (nonnull SEIdFaceSimilarityResult *) compareFacesFromDocument:(nonnull SECommonImageRef *)photo
                                                      andSelfie:(nonnull SECommonImageRef *)image;

- (nonnull SEIdFaceDescription *) getDescriptionFromImage:(nonnull SECommonImageRef *)image;
    

@end

